/// <reference types="lib.es2017.observable" />
import { DeviceFilter as FullDeviceFilter } from '@signageos/common-types/dist/Device/deviceFilter';
import IChanges from '../../IChanges';
import { OffsetPagination, PaginationAndSorting, TPagination } from '../../Lib/Pagination/paginationTypes';
import { TSortOrder } from '../../Lib/Sort/sort';
import { IAccount } from '../Account/accountModel';
import { IOrganization, IOrganizationIdentification, SubscriptionType, IDistributionByDevicePlan, OrganizationsDeviceCountsStatisticsType } from '../Organization/organizationModel';
import { IApplication } from '../Application/applicationModel';
import ApplicationType from '@signageos/common-types/dist/Application/ApplicationType';
import { IPolicyIdentification, IPolicyWithPriority } from '../Policy/policyModel';
import { LocationIdentification } from '../Location/locationModel';
import { IOrganizationTag } from '../Organization/Tag/organizationTagModel';
import { ValidateExact } from '../../TypeScript/types';
/**
 * The interface that represents and partial object containing only the artificial identification of a specific device.
 * In this artificial identification, the identityHash is the only required property.
 */
export interface IDeviceArtificialIdentification {
    /**
     * Unlike the `uid` property, this property is generated automatically by the server when the device entity is created.
     * Usually, the entity has it's artificial identification property called `uid`, however because of historical reasons
     * the property in IDevice entity is called `identityHash` instead.
     * So the spirit of this identification `identityHash` is same as all other entities `uid` property.
     * This `identityHash` is the preferred property to be used for all inter-service communication instead of `uid`.
     * This property is also used as a public identifier in REST API and in UI for end users.
     */
    identityHash: string;
}
/**
 * The interface that represents and partial object containing only the physical identification of a specific device.
 * In this physical identification, the uid is the only required property.
 */
export interface IDevicePhysicalIdentification {
    /**
     * Unlike the `identityHash` property, this property is generated by the device itself.
     * It's usually based on some kind of internal physical identification of the device. E.g.: serial number, MAC address, CPU ID, etc.
     * So for a single device the `uid` property has to be always computed deterministically same. Even if the device is removed
     * from the production database and the `identityHash` is lost or changed, the `uid` property has to be always the same as before.
     * This `uid` is usually called differently outside the IDevice entity context. The possible variations are: `duid`, `DUID`, `deviceUid`.
     * Never interchange the `uid` property with `identityHash`.
     * You have to be always careful of what identification are you using in which situation.
     */
    uid: string;
}
/**
 * This object is just combination of both identifiers of IDevice entity.
 * For some reasons it's easier to have available both identifiers at the same time.
 */
export interface IDeviceIdentification extends IDeviceArtificialIdentification, IDevicePhysicalIdentification {
}
export interface IDevice extends IDeviceIdentification {
    name?: string | null;
    /** Used mainly for sorting devices by name case insensitive in database */
    nameLowerCase?: string | null;
    model?: string | null;
    serialNumber?: string | null;
    brand?: string | null;
    osVersion?: string | null;
    applicationType: ApplicationType;
    createdAt: Date;
    organizationUid?: string | null;
    /** @deprecated use deviceConfigurationModel entity instead */
    bannedSince?: Date | null;
    nameNormalized?: string | null;
    accountId?: number | null;
    pinCode?: string | null;
    /**
     * This property should have been available in all new devices since specific front-display version (see changelog).
     * For old devices, there is not available. Allow any firmware to be installed on the device, it will fail during installation anyways.
     * Example values: rpi, rpi4, benq_sl490
     */
    firmwareType?: string;
    /** Specific firmware version currently installed on device (some old device could not report it) */
    firmwareVersion?: string | null;
    /** @deprecated use deviceConfigurationModel entity instead */
    subscriptionType?: SubscriptionType | null;
    tagUids?: string[];
    policies?: IPolicyWithPriority[] | null;
    lastOrganizationUid?: string | null;
    /**
     * Date of device provision (same as `deviceVerification.verifiedAt`)
     * however it cannot be set back to undefined
     */
    lastProvisionAt?: Date;
    /**
     * Date of device deprovision
     * @note if lastDeprovisionAt > lastProvisionAt then device is deprovisioned
     * @note if lastDeprovisionAt < lastProvisionAt then device is provisioned
     */
    lastDeprovisionAt?: Date;
    /**
     * If set to any date, the device is connected to remote control server (example tizen magic info server).
     * This device has more management features to perform.
     */
    extendedManagementSince?: Date | null;
    /**
     * Code which will be used for pairing devices with extended management with organization/company or other entity
     */
    extendedManagementPairCode?: string | null;
    /**
     * The very first date when device was provisioned
     * Never should be changed after it was set for the first time
     */
    firstProvisionedAt?: Date;
    /**
     * Location can have multiple devices. Each device can be assigned to only one location
     */
    locationUid?: string;
}
export declare type IDeviceMinimal = Pick<IDevice, TDeviceRequired>;
export declare type TDeviceRequired = 'uid' | 'identityHash' | 'organizationUid';
export declare type FieldSorterType = 'applicationType' | 'name' | 'model' | 'firmwareVersion' | 'firmwareType' | 'serialNumber' | 'createdAt' | 'lastDeprovisionAt';
export interface IDevicePropertySorter {
    field: FieldSorterType;
    direction: TSortOrder;
}
export declare type CountByProperty = keyof Pick<IDevice, 'organizationUid' | 'applicationType' | 'locationUid'>;
export declare type CountByPropertyType<K extends CountByProperty> = Required<NonNullable<IDevice[K]>>;
export interface IFetchCountMapByPropertyParams<K extends CountByProperty, TDeviceFilter extends FullDeviceFilter = DefaultDeviceFilter> {
    organizations: IOrganization[];
    property: K;
    propertyFilter?: TDeviceFilter;
}
export interface IDeviceCSVExportable extends IDevice {
    productionOrTest: 'test' | 'production';
    organizationName: string;
    companyName: string;
}
/** By default, the only identityHash is supported in DeviceFilter. At least one property is required and identityHash is uniq identifier */
export declare type DefaultDeviceFilter = Pick<FullDeviceFilter, 'identityHash'>;
/**
 * @see IDeviceModelRead
 */
export declare type IDeviceModel<TDevice extends IDevice, TDeviceFilter extends FullDeviceFilter = DefaultDeviceFilter> = IDeviceModelRead<TDevice, TDeviceFilter> & IDeviceModelWrite<TDevice> & IDeviceModelObservable<TDevice>;
/**
 * The second generic parameter is used to filter out properties from DeviceFilter which are not supported by the model.
 * Different implementations of the model may support different properties. Based on the given dependencies.
 * By default only 'identityHash' property in filter are supported.
 */
export interface IDeviceModelRead<TDevice extends IDevice, DeviceFilter extends FullDeviceFilter = DefaultDeviceFilter> {
    getDeviceIterator(): AsyncIterableIterator<TDevice>;
    getListByOrganizationsIterator<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<AsyncIterableIterator<TDevice>>;
    fetchListByAccount(account: IAccount, ignoreOrganizations?: boolean): Promise<TDevice[]>;
    fetchByUid(uid: string): Promise<TDevice | null>;
    fetchByIdentityHash(identityHash: string): Promise<TDevice | null>;
    fetchListByIdentityHashes(identityHashes: string[], offset?: number, limit?: number): Promise<TDevice[]>;
    /**
     * @search cam't be combined with fields param Due conflicting projection. Mongo doesn't allow to have non homogenous $project.
     * @fields can't be combined with filter param (exactly with filter.search). Due conflicting projection.
     * Mongo doesn't allow to have non homogenous $project.
     */
    fetchListByOrganizations<F extends keyof IDevice, Filter = DeviceFilter>(organizations: IOrganization[], filter?: ValidateExact<Filter, DeviceFilter>, sorter?: IDevicePropertySorter, offset?: OffsetPagination['offset'], limit?: OffsetPagination['limit'], fields?: F[]): Promise<Pick<IDevice, F extends keyof IDeviceMinimal ? F | TDeviceRequired : keyof IDevice>[]>;
    /** @deprecated use fetchCountMapByProperty method instead */
    countByOrganizations<Filter = DeviceFilter>(organizations: IOrganization[], filter?: ValidateExact<Filter, DeviceFilter>): Promise<number>;
    /** @deprecated use fetchCountMapByProperty method instead */
    countMapOfOrganizations<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<Map<string, number>>;
    /** @deprecated use fetchCountMapByProperty method instead */
    countMapOfApplicationTypes<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<Map<ApplicationType, number>>;
    countMapOfLocations<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<Map<string, number>>;
    fetchCountMapByProperty<K extends CountByProperty, Filter = DeviceFilter>(fetchParams: IFetchCountMapByPropertyParams<K, ValidateExact<Filter, DeviceFilter>>): Promise<Map<CountByPropertyType<K>, number>>;
    fetchIdentityHashesByOrganizations<Filter = DeviceFilter>(organizations: IOrganization[], filter?: ValidateExact<Filter, DeviceFilter>): Promise<string[]>;
    fetchGroupedIdentificationsByOrganizations<Filter = DeviceFilter>(organizations: IOrganization[], filter?: ValidateExact<Filter, DeviceFilter>): Promise<{
        [orgUid: string]: IDeviceIdentification[];
    }>;
    fetchListByPropertyFilter<F extends keyof IDevice, Filter = DeviceFilter>(propertyFilter: ValidateExact<Filter, DeviceFilter>, limit?: number, fields?: F[]): Promise<Pick<IDevice, F extends keyof IDevice ? F | TDeviceRequired : keyof IDevice>[]>;
    /**
     * Returns an array of UID pairs. The object contains identityHash and uid (sometimes called DUID)
     */
    fetchUidListByPropertyFilter<Filter = DeviceFilter>(propertyFilter: ValidateExact<Filter, DeviceFilter>, limit?: number): Promise<IDeviceIdentification[]>;
    /**
     * Returns an array of UID pairs. The object contains identityHash and uid (sometimes called DUID)
     */
    fetchUidListByOrganizationUids<Filter = DeviceFilter>(organizationUids: string[], propertyFilter: ValidateExact<Filter, DeviceFilter>, limit?: number): Promise<IDeviceIdentification[]>;
    countByPropertyFilter<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter: ValidateExact<Filter, DeviceFilter>, pagination?: TPagination): Promise<number>;
    fetchModelsByOrganizations<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<string[]>;
    /**
     * Returns list of applicationType types which are present in devices for given organizations
     * @param organizations - array of organizations
     * @param propertyFilter - property filter to filter out devices
     */
    fetchApplicationTypesByOrganizations<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<ApplicationType[]>;
    fetchAllFirmwareTypes(applicationType: ApplicationType): Promise<string[]>;
    fetchFirmwareVersionsByOrganizations<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<string[]>;
    fetchDeprovisionedSince(organizations: IOrganization[], since: Date, paginationAndSorting?: PaginationAndSorting<FieldSorterType>): Promise<TDevice[]>;
    getDeprovisionedCountByOrganizations(organizations: IOrganization[], since?: Date, before?: Date): Promise<number>;
    fetchDevicesDistributionByDevicePlan<Filter = DeviceFilter>(provisionedSince?: Date, provisionedBefore?: Date, isProduction?: boolean, propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<IDistributionByDevicePlan>;
    fetchOrganizationDistributionByDevicePlan(organizationUid: string, provisionedSince?: Date, provisionedBefore?: Date): Promise<IDistributionByDevicePlan>;
    fetchOrganizationsDistributionByDevicePlan<Filter = DeviceFilter>(organizationUids: string[], createdSince?: Date, createdBefore?: Date, propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<OrganizationsDeviceCountsStatisticsType>;
    fetchListByCompanyUids(companyUids: string[]): Promise<IDeviceCSVExportable[]>;
    fetchByPolicy(policy: IPolicyWithPriority): Promise<TDevice[]>;
    countByPolicy(policy: IPolicyWithPriority): Promise<number>;
    countByPolicyAndApplicationType(policy: IPolicyWithPriority, applicationType: ApplicationType): Promise<number>;
    fetchOrganizationsUidsDevicesBelongTo<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter: ValidateExact<Filter, DeviceFilter>): Promise<(string | null)[]>;
    fetchTags(identityHash: TDevice['identityHash'], paginationAndSorting?: PaginationAndSorting<'createdAt'>): Promise<IOrganizationTag[]>;
    fetchOsVersionsByOrganizations<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<NonNullable<IDevice['osVersion']>[]>;
    fetchBrandsByOrganizations<Filter = DeviceFilter>(organizations: IOrganization[], propertyFilter?: ValidateExact<Filter, DeviceFilter>): Promise<NonNullable<IDevice['brand']>[]>;
    fetchCreatedBetween(args: {
        createdSince?: Date;
        createdUntil: Date;
    }): Promise<TDevice[]>;
    fetchListOfLocationUidsByDeviceIdentityHashes(identityHashes: TDevice['identityHash'][]): Promise<string[]>;
}
export interface IDeviceModelWrite<TDevice extends IDevice> {
    create(uid: string, identityHash: string, application: IApplication, createdAt: Date): Promise<void>;
    updateModel(deviceRow: TDevice, model: string): Promise<void>;
    updateSerialNumber(deviceRow: TDevice, serialNumber: string): Promise<void>;
    updateBrand(deviceRow: TDevice, brand: string): Promise<void>;
    updateOSVersion(deviceRow: TDevice, osVersion: string): Promise<void>;
    updateName(deviceRow: TDevice, name: string): Promise<void>;
    updateExtendedManagementSince(deviceRow: TDevice, extendedManagementSince: Date | null): Promise<void>;
    updateExtendedManagementPairCode(deviceRow: TDevice, extendedManagementPairCode: string): Promise<void>;
    updateOrganization(deviceRow: TDevice, organization: IOrganizationIdentification): Promise<void>;
    removeOrganization(deviceRow: TDevice): Promise<void>;
    /** @deprecated use deviceConfigurationModel instead */
    markBanned(deviceRow: TDevice, bannedSince: Date): Promise<void>;
    /** @deprecated use deviceConfigurationModel instead */
    markApproved(deviceRow: TDevice, approvedSince: Date): Promise<void>;
    /** @deprecated use deviceConfigurationModel instead */
    changeSubscriptionType(deviceRow: TDevice, subscriptionType: SubscriptionType | null): Promise<void>;
    updateAccount(deviceRow: TDevice, account: IAccount): Promise<void>;
    removeAccount(deviceRow: TDevice): Promise<void>;
    updatePinCode(deviceRow: TDevice, pinCode: string): Promise<void>;
    updateFirmwareType(deviceRow: TDevice, firmwareType: string): Promise<void>;
    updateFirmwareVersion(deviceRow: TDevice, firmwareVersion: string): Promise<void>;
    assignTag(deviceRow: TDevice, tagUid: string): Promise<void>;
    removeTag(deviceRow: TDevice, tagUid: string): Promise<void>;
    assignOrUpdatePolicy(deviceRow: TDevice, policy: IPolicyWithPriority): Promise<void>;
    removePolicy(deviceRow: TDevice, policy: IPolicyIdentification): Promise<void>;
    setLastProvisionedAt(deviceRow: TDevice, provisionedAt?: Date): Promise<void>;
    setLastDeprovisionedAt(deviceRow: TDevice, deprovisionedAt?: Date): Promise<void>;
    setFirstProvisionedAt(deviceRow: TDevice, provisionedAt: Date): Promise<void>;
    assignDeviceToLocation(deviceRow: TDevice, location: LocationIdentification): Promise<void>;
    unassignDeviceFromLocation(deviceRow: TDevice): Promise<void>;
}
export interface IDeviceModelObservable<T extends IDevice> {
    observeListByIdentityHashes(identityHashes: string[]): Promise<Observable<IChanges<T>, Error>>;
    observeListByAccount(account: IAccount, ignoreOrganizations?: boolean): Promise<Observable<IChanges<T>, Error>>;
    observeByIdentityHash(identityHash: string): Promise<Observable<IChanges<T>, Error>>;
}
//# sourceMappingURL=deviceModel.d.ts.map