/// <reference types="lib.es2017.observable" />
import { ICursor } from '../../Cursor/ICursor';
import { IOriginator } from '../Originator/originator';
import { IOrganization } from '../Organization/organizationModel';
import IChanges from '../../IChanges';
import { NotificationOptions } from '@signageos/common-types/dist/Alert/NotificationOptions';
export interface IHistoryItem<A extends IAlertRule> {
    /** Newly set values. Empty values were not changed */
    values: HistoryItemValues<A>;
    /** Date of update in history */
    updatedAt: Date;
    originator: IOriginator;
}
/**
 * @external AlertRule in @signageos/common/devices
 * Empty (undefined) values means that rule is not filled and is not valid to be checked.
 */
export interface IAlertRule {
    /** can't be changed after the creation thus isn't saved into history */
    alertRuleUid: string;
    alertType?: string | null;
    name: string;
    companyUid: string;
    /** can't be changed after the creation thus isn't saved into history */
    createdAt: Date;
    description?: string | null;
    organizationUids?: string[] | null;
    filter?: unknown | null;
    conditions?: unknown | null;
    threshold?: unknown | null;
    periodicity?: unknown | null;
    action?: unknown | null;
    /** contains all user interactions with alert rule */
    history: IHistoryItem<IAlertRule>[];
    archivedAt: Date | null;
    pausedAt: Date | null;
    notificationOptions?: NotificationOptions | null;
}
declare type AllUpdatable<A extends IAlertRule> = Pick<A, keyof Omit<IAlertRule, 'alertRuleUid' | 'createdAt'>>;
export declare type Updatable<A extends IAlertRule> = Partial<Omit<AllUpdatable<A>, 'archivedAt' | 'pausedAt'>>;
export declare type HistoryItemValues<A extends IAlertRule> = Partial<Omit<AllUpdatable<A>, 'history'>>;
export declare type IAlertRuleModel<A extends IAlertRule> = IAlertRuleModelRead<A> & IAlertRuleModelWrite<A>;
export declare type FieldSorterType = 'name' | 'alertType' | 'createdAt';
export interface IAlertRulePropertySorter {
    field: FieldSorterType;
    direction: 'ascending' | 'descending';
}
export interface IAlertRuleFetchOptions<A extends IAlertRule> {
    /** Sorting out output */
    sorter?: IAlertRulePropertySorter;
    /** Filter by properties */
    filter?: {
        /** Required filter for fetching according to company */
        companyUid?: string;
        /** Required filter to limit organizations scope */
        organizations?: IOrganization[];
        /** Exact matching of alertType */
        alertTypes?: A['alertType'][];
        /** Query by name even partial string */
        name?: string;
        isArchived?: boolean;
        isPaused?: boolean;
        createdUntil?: Date;
    };
    /** limiting and offsetting */
    pagination?: {
        offset?: number;
        limit?: number;
    };
}
export interface IAlertRuleStatusFilter {
    includeActive?: boolean;
    includePaused?: boolean;
    includeArchived?: boolean;
}
export interface IAlertRuleModelRead<A extends IAlertRule> {
    fetchByUid(alertRuleUid: string): Promise<A | null>;
    fetchListByOptions(options: IAlertRuleFetchOptions<A>): Promise<A[]>;
    fetchListByCompanyUidsAndOptions(companyUids: string[], options: IAlertRuleFetchOptions<A>, status?: IAlertRuleStatusFilter): Promise<A[]>;
    fetchActiveCursorExcludedUids(excludedAlertRuleUids: string[]): Promise<ICursor<A>>;
    countByCompanyUidsAndOptions(companyUids: string[], options: IAlertRuleFetchOptions<A>, status?: IAlertRuleStatusFilter): Promise<number>;
}
export interface IAlertRuleModelWrite<A extends IAlertRule> {
    create(alertRuleUid: string, name: string, companyUid: string, createdAt: Date, originator: IOriginator): Promise<void>;
    update(alertRuleRow: A, values: Updatable<A>, updatedAt: Date, originator: IOriginator): Promise<void>;
    archive(alertRuleRow: A, archivedAt: Date, originator: IOriginator): Promise<void>;
    pause(alertRuleRow: A, pausedAt: Date, originator: IOriginator): Promise<void>;
    unpause(alertRuleRow: A, unpausedAt: Date, originator: IOriginator): Promise<void>;
}
export interface IAlertRuleModelObservable<A extends IAlertRule> {
    observeListByCompanyUidsAndOptions(companyUids: string[], options: IAlertRuleFetchOptions<A>, status?: IAlertRuleStatusFilter): Promise<Observable<A[], Error>>;
    observeByUid(alertRuleUid: string): Promise<Observable<IChanges<A>, Error>>;
}
export {};
//# sourceMappingURL=alertRuleModel.d.ts.map