/// <reference types="lib.es2017.observable" />
import { IOriginator } from '../Originator/originator';
import { IOrganization } from '../Organization/organizationModel';
import { IAlertRule } from './alertRuleModel';
import IChanges from '../../IChanges';
export interface IHistoryItem {
    /** Newly set values. Empty values were not changed */
    values: {
        description?: string;
        snoozeRule?: unknown | null;
        archivedAt?: Date | null;
        snoozedDevice?: SnoozedDevice;
        unsnoozedDevice?: {
            deviceIdentityHash: string;
            unsnoozedAt: Date;
        };
        unassignedDevice?: {
            deviceIdentityHash: string;
            unassignedAt: Date;
        };
    };
    /** Date of update in history */
    updatedAt: Date;
    originator: IOriginator;
}
declare type SnoozedDevice = {
    deviceIdentityHash: string;
    snoozeRule: unknown;
};
/** @external Alert in @signageos/common/devices */
export interface IAlert {
    alertUid: string;
    organizationUid: string;
    description: string;
    alertRuleUid: string;
    createdAt: Date;
    /** Is updated only when assigned devices has changed */
    latelyChangedAt: Date;
    /** Is updated every time alert is checked */
    latelyProcessedAt?: Date;
    deviceIdentityHashes: string[];
    /** contains all user interactions with alert */
    history: IHistoryItem[];
    snoozeRule: unknown | null;
    archivedAt: Date | null;
    snoozedDevices?: SnoozedDevice[];
}
export interface IDeviceToAlertsRelation {
    deviceIdentityHash: string;
    alertUids: string[];
}
export interface IAlertUidWithStats {
    alertUid: string;
    affectedDevices: number;
}
export interface ILocationToAlertsRelation {
    locationUid: string;
    activeAlerts: IAlertUidWithStats[];
}
declare type OwnFieldSorterType = keyof Pick<IAlert, 'description' | 'deviceIdentityHashes' | 'createdAt' | 'latelyChangedAt'>;
declare type ForeignSorterFields = 'organizationTitle';
declare type FieldSorterType = OwnFieldSorterType | ForeignSorterFields;
export interface IAlertListSorter {
    field: FieldSorterType;
    direction: 'ascending' | 'descending';
}
export declare enum OrderDirection {
    ASC = 1,
    DESC = -1
}
export interface IAlertListPagination {
    offset?: number;
    limit?: number;
}
export interface IAlertFetchOptions {
    orderBy?: {
        createdAt?: OrderDirection;
    };
    filter?: {
        organization: IOrganization;
        isArchived?: boolean;
        isSnoozed?: boolean;
        createdUntil?: Date;
    };
    /** limiting and offsetting */
    pagination?: IAlertListPagination;
}
export interface IAlertListPropertyFilter {
    includeActive?: boolean;
    includeArchived?: boolean;
    includeSnoozed?: boolean;
    alertTypes?: string[];
    description?: string;
    hasDevices?: boolean;
}
export declare type IAlertModel<A extends IAlert> = IAlertModelRead<A> & IAlertModelWrite<A>;
export interface IAlertModelRead<A extends IAlert> {
    fetchByUid(alertUid: string): Promise<A | null>;
    fetchListByOrganization(options: IAlertFetchOptions): Promise<A[]>;
    fetchListByOrganizations(organizations: IOrganization[], filter?: IAlertListPropertyFilter, sorter?: IAlertListSorter, pagination?: IAlertListPagination): Promise<A[]>;
    fetchListByOrganizationsAndDeviceIdentityHashes(organizations: IOrganization[], deviceIdentityHashes: string[]): Promise<A[]>;
    fetchLastActiveByOrganizationUidAndRule(organizationUid: string, alertRule: IAlertRule): Promise<A | null>;
    countByOrganizations(organizations: IOrganization[], filter?: IAlertListPropertyFilter): Promise<number>;
    countByDeviceIdentityHash(deviceIdentityHash: string, filter?: IAlertListPropertyFilter): Promise<number>;
    countByDeviceIdentityHashes(deviceIdentityHashes: string[], filter?: IAlertListPropertyFilter): Promise<number>;
    countByOrganizationAndDeviceIdentityHashes(organization: IOrganization, deviceIdentityHashes: string[], filter?: IAlertListPropertyFilter): Promise<number>;
    fetchListByDeviceIdentityHash(deviceIdentityHash: string, filter?: IAlertListPropertyFilter): Promise<A[]>;
    countDevicesWithActiveAlertsByOrganizations(organizations: IOrganization[], filter?: IAlertListPropertyFilter): Promise<number>;
    fetchDevicesWithActiveAlertsByOrganizations(organizations: IOrganization[], filter?: IAlertListPropertyFilter): Promise<IDeviceToAlertsRelation[]>;
    countLocationsWithActiveAlertsByOrganizations(organizations: IOrganization[], filter?: IAlertListPropertyFilter): Promise<number>;
    fetchLocationsWithActiveAlertsByOrganizations(organizations: IOrganization[], filter?: IAlertListPropertyFilter): Promise<ILocationToAlertsRelation[]>;
}
export interface IAlertModelWrite<A extends IAlert> {
    create(alertUid: string, organization: IOrganization, description: string, alertRule: IAlertRule, createdAt: Date): Promise<void>;
    updateDescription(alertRow: A, description: string, updatedAt: Date, originator: IOriginator): Promise<void>;
    updateLatelyProcessedAt(alertRow: A, latelyProcessedAt: Date): Promise<void>;
    assignDevice(alertRow: A, deviceIdentityHash: string, assignedAt: Date): Promise<void>;
    /**
     *
     * @description when `originator` is not provided, it is assumed that the action was performed by the system and no history is created
     */
    unassignDevice(alertRow: A, deviceIdentityHash: string, unassignedAt: Date, originator?: IOriginator): Promise<void>;
    archive(alertRow: A, archivedAt: Date, originator: IOriginator): Promise<void>;
    unarchive(alertRow: A, unarchivedAt: Date, originator: IOriginator): Promise<void>;
    snooze(alertRow: A, snoozeRule: A['snoozeRule'], snoozedAt: Date, originator: IOriginator): Promise<void>;
    unsnooze(alertRow: A, unsnoozedAt: Date, originator: IOriginator): Promise<void>;
    snoozeDevice(alertRow: A, deviceIdentityHash: string, snoozeRule: A['snoozeRule'], snoozedAt: Date, originator: IOriginator): Promise<void>;
    unsnoozeDevice(alertRow: A, deviceIdentityHash: string, unsnoozedAt: Date, originator: IOriginator): Promise<void>;
}
export interface IAlertModelObservable<T extends IAlert> {
    observeListByOrganizations(organizations: IOrganization[], filter?: IAlertListPropertyFilter, sorter?: IAlertListSorter, pagination?: IAlertListPagination): Promise<Observable<T[], Error>>;
    observeByUid(alertUid: string): Promise<Observable<IChanges<T>, Error>>;
}
export {};
//# sourceMappingURL=alertModel.d.ts.map