/// <reference types="lib.es2017.observable" />
import { AccessLevel, Role } from '@signageos/common-types/dist/Account/Account';
import IChanges from '../../IChanges';
import { PaginationAndSorting } from '../../Lib/Pagination/paginationTypes';
export declare enum PrivilegeEntity {
    Company = "company",
    Organization = "organization"
}
export interface IPrivilege {
    entity: PrivilegeEntity;
    entityUid: string;
    role: Role;
    assignedAt: Date;
}
export interface ISession {
    sessionUid: string;
    createdAt: Date;
}
export interface IAccount {
    id: number;
    username: string;
    email: string;
    firstname?: string;
    lastname?: string;
    accessLevel: AccessLevel;
    createdAt: Date;
    sessions: ISession[];
    password?: string;
    salt?: string;
    isActive: boolean;
    privileges: IPrivilege[];
}
export interface IAccountListPagination {
    offset?: number;
    limit?: number;
}
export declare enum OrderDirection {
    ASC = 1,
    DESC = -1
}
export interface IAccountListSorter {
    uid?: OrderDirection;
    name?: OrderDirection;
    email?: OrderDirection;
    created?: OrderDirection;
}
export interface IAccountPropertyFilter {
    ids?: number[];
    fulltext?: string;
    type?: string;
    /**
     * There is OR condition between organizationUids and accountsWithoutOrganization filters
     * User can filter accounts with assigned organizations & accounts without organization
     */
    /**
     * Passed array has to be truly organizations uids.
     * This filter property doesn´t work for companies uids because filter consider
     * only privileges with privilege entity equals to @type {PrivilegeEntity.Organization}.
     */
    organizationUids?: string[];
    /**
     * Return accounts with/without organizations
     */
    accountsWithoutOrganization?: boolean;
    rolesInCompany?: {
        companyUid: string;
        roles: Role[];
    };
    emailDomains?: string[];
    excludeAdmins?: boolean;
}
export interface IAccountFetchOptions {
    filter?: IAccountPropertyFilter;
    sorter?: IAccountListSorter;
    pagination?: IAccountListPagination;
}
export declare type IAccountModel<T extends IAccount> = IAccountModelRead<T> & IAccountModelWrite<T> & IAccountModelObservable<T>;
export declare type AccountPrivilege<TAccount extends IAccount> = {
    account: TAccount;
    privilege: TAccount['privileges'][0];
};
export interface CreateAccountParams {
    id: number;
    username: string;
    email: string;
    firstname?: string;
    lastname?: string;
    createdAt: Date;
    accessLevel: AccessLevel;
}
export interface IAccountModelRead<TAccount extends IAccount> {
    fetchById(id: number): Promise<TAccount | null>;
    fetchBySessionUid(sessionUid: string): Promise<TAccount | null>;
    fetchByUsername(username: string, skipArchived?: boolean): Promise<TAccount | null>;
    fetchByEmail(email: string, skipArchived?: boolean): Promise<TAccount | null>;
    /**
     * This method is used only when we want to find the correct account
     * for case-sensitive email duplicates, eg. joe.doe@domain.com vs. Joe.Doe@domain.com.
     * We should strongly avoid these situations and they should never happen.
     * That is why the `fetchByEmail` method is recommended for regular cases.
     * @deprecated Use fetchByEmail instead */
    fetchByEmailCaseSensitive(email: string): Promise<TAccount | null>;
    fetchListByIds(ids: number[]): Promise<TAccount[]>;
    fetchFilteredList(filter?: IAccountPropertyFilter, pagination?: IAccountListPagination): Promise<TAccount[]>;
    fetchAllList(): Promise<TAccount[]>;
    fetchSortedList(options: IAccountFetchOptions): Promise<TAccount[]>;
    fetchTotalCount(filter?: IAccountPropertyFilter): Promise<number>;
    fetchAllCompanyAccounts(companyUid: string): Promise<TAccount[]>;
    fetchPrivilegesByOrganizationUid(organizationUid: string, accountId?: TAccount['id'], paginationAndSorting?: PaginationAndSorting<'privilege.assignedAt'>): Promise<AccountPrivilege<TAccount>[]>;
}
export interface IAccountModelWrite<TAccount extends IAccount> {
    create(params: CreateAccountParams): Promise<void>;
    updateEmail(accountRow: TAccount, email: string): Promise<void>;
    updatePasswordAndSalt(accountRow: TAccount, password: string, salt: string): Promise<void>;
    updateAccessLevel(accountRow: TAccount, accessLevel: AccessLevel): Promise<void>;
    updateFirstNameAndLastName(accountRow: TAccount, firstname: string, lastname: string): Promise<void>;
    setActive(accountRow: TAccount, active: boolean): Promise<void>;
    addSession(accountRow: TAccount, sessionUid: string, createdAt: Date): Promise<void>;
    removeSession(accountRow: TAccount, sessionUid: string): Promise<void>;
    removeAllSessions(accountRow: TAccount): Promise<void>;
    addPrivilege(accountRow: TAccount, privilege: IPrivilege): Promise<void>;
    removePrivilege(accountRow: TAccount, privilege: IPrivilege): Promise<void>;
}
export interface IAccountModelObservable<TAccount extends IAccount> {
    observeById(id: number): Promise<Observable<IChanges<TAccount>, Error>>;
    observeAllList(): Promise<Observable<IChanges<TAccount>, Error>>;
    observeSortedList(options: IAccountFetchOptions): Promise<Observable<IAccount[], Error>>;
    waitByUsername(username: string): Promise<TAccount>;
    waitBySessionUid(sessionUid: string): Promise<TAccount>;
}
//# sourceMappingURL=accountModel.d.ts.map