"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.increment = exports.exists = exports.set = exports.countOrderedSet = exports.addToOrderedSet = exports.scanKeysByPattern = exports.CURSOR_BEGINNING = exports.observe = exports.updateOne = exports.updateOnDelete = exports.updatePartialOne = exports.getKeysByPattern = exports.getMoreByPattern = exports.flushAll = exports.getManyMapWithKeys = exports.getMany = exports.getAllOfHash = exports.getValuesOfHash = exports.getOne = exports.createAndIncrementInHash = exports.deleteOneFromHash = exports.deleteOne = exports.setExpiration = exports.createAndAddToHash = exports.createOneWithExpiration = exports.createOne = void 0;
const Debug = require("debug");
const jsonHelper_1 = require("@signageos/lib/dist/JSON/jsonHelper");
const wait_1 = require("@signageos/lib/dist/Timer/wait");
const bson_1 = require("bson");
const debug = Debug('@signageos/user-domain-model:Redis:helper');
function createOne(conn, key, value) {
    const message = stringifyDocument(value);
    return new Promise((resolve, reject) => {
        conn.connection.set(key, message, (error) => {
            if (error) {
                reject(error);
            }
            else {
                resolve();
            }
        });
    });
}
exports.createOne = createOne;
function createOneWithExpiration(conn, key, value, ttlSeconds) {
    const message = stringifyDocument(value);
    return new Promise((resolve, reject) => {
        conn.connection.setex(key, ttlSeconds, message, (error) => {
            if (error) {
                reject(error);
            }
            else {
                resolve();
            }
        });
    });
}
exports.createOneWithExpiration = createOneWithExpiration;
function createAndAddToHash(conn, key, field, value) {
    const message = stringifyDocument(value);
    return new Promise((resolve, reject) => {
        conn.connection.hset(key, field, message, (error) => {
            if (error) {
                reject(error);
            }
            else {
                resolve();
            }
        });
    });
}
exports.createAndAddToHash = createAndAddToHash;
function setExpiration(conn, key, ttlSeconds) {
    return new Promise((resolve, reject) => {
        conn.connection.expire(key, ttlSeconds, (error) => {
            if (error) {
                reject(error);
            }
            else {
                resolve();
            }
        });
    });
}
exports.setExpiration = setExpiration;
function deleteOne(conn, key) {
    return new Promise((resolve, reject) => {
        conn.connection.del(key, (error) => {
            if (error) {
                reject(error);
            }
            else {
                resolve();
            }
        });
    });
}
exports.deleteOne = deleteOne;
function deleteOneFromHash(conn, key, field) {
    return new Promise((resolve, reject) => {
        conn.connection.hdel(key, field, (error) => {
            if (error) {
                reject(error);
            }
            else {
                resolve();
            }
        });
    });
}
exports.deleteOneFromHash = deleteOneFromHash;
function createAndIncrementInHash(conn, key, field, incrementAmount) {
    return new Promise((resolve, reject) => {
        conn.connection.hincrby(key, field, incrementAmount, (error) => {
            if (error) {
                reject(error);
            }
            else {
                resolve();
            }
        });
    });
}
exports.createAndIncrementInHash = createAndIncrementInHash;
function getOne(conn, key) {
    return new Promise((resolve, reject) => {
        conn.connection.get(key, (error, message) => {
            if (error) {
                reject(error);
            }
            else {
                resolve(parseDocument(message));
            }
        });
    });
}
exports.getOne = getOne;
function getValuesOfHash(conn, key) {
    return new Promise((resolve, reject) => {
        conn.connection.hvals(key, (error, messages) => {
            if (error) {
                reject(error);
            }
            else {
                const documents = [];
                messages.forEach((msg) => {
                    const parsed = parseDocument(msg);
                    if (parsed) {
                        documents.push(parsed);
                    }
                });
                resolve(documents);
            }
        });
    });
}
exports.getValuesOfHash = getValuesOfHash;
function getAllOfHash(conn, key) {
    return new Promise((resolve, reject) => {
        conn.connection.hgetall(key, (error, messages) => {
            if (error) {
                reject(error);
            }
            else {
                resolve(messages);
            }
        });
    });
}
exports.getAllOfHash = getAllOfHash;
function getMany(conn, keys) {
    // mget can't be called with empty array
    if (keys.length === 0) {
        return Promise.resolve([]);
    }
    return new Promise((resolve, reject) => {
        conn.connection.mget(keys, (error, messages) => {
            if (error) {
                reject(error);
            }
            else {
                const result = [];
                messages.forEach((message) => {
                    if (message) {
                        const p = parseDocument(message);
                        if (p) {
                            result.push(p);
                        }
                    }
                });
                resolve(result);
            }
        });
    });
}
exports.getMany = getMany;
function getManyMapWithKeys(conn, keys) {
    // mget can't be called with empty array
    if (keys.length === 0) {
        return Promise.resolve(new Map());
    }
    return new Promise((resolve, reject) => {
        conn.connection.mget(keys, (error, messages) => {
            if (error) {
                reject(error);
            }
            else {
                const result = new Map();
                messages.forEach((message, index) => {
                    if (message) {
                        const p = parseDocument(message);
                        if (p) {
                            result.set(keys[index], p);
                        }
                    }
                });
                resolve(result);
            }
        });
    });
}
exports.getManyMapWithKeys = getManyMapWithKeys;
function flushAll(conn) {
    return new Promise((resolve) => {
        conn.connection.flushall(() => {
            resolve();
        });
    });
}
exports.flushAll = flushAll;
function getMoreByPattern(conn, keyPattern) {
    return __awaiter(this, void 0, void 0, function* () {
        const keys = yield getKeysByPattern(conn, keyPattern);
        const documents = yield Promise.all(keys.map((key) => getOne(conn, key)));
        const stillExistingDocuments = documents.filter((document) => document !== null);
        return stillExistingDocuments;
    });
}
exports.getMoreByPattern = getMoreByPattern;
/**
 * Get all keys matching the pattern. It uses KEYS which is blocking and memory pressured variant.
 * Consider using scanKeysByPattern instead.
 */
function getKeysByPattern(conn, keyPattern) {
    return new Promise((resolve, reject) => {
        conn.connection.keys(keyPattern, (error, keys) => {
            if (error) {
                reject(error);
            }
            else {
                resolve(keys);
            }
        });
    });
}
exports.getKeysByPattern = getKeysByPattern;
function stringifyDocument(document) {
    return JSON.stringify(document);
}
function parseDocument(message) {
    return JSON.parse(message, jsonHelper_1.deserializeJSON);
}
function updatePartialOne(conn, key, value) {
    const valueString = stringifyChanges({ new_partial_val: value });
    debug('updatePartialOne', key, valueString);
    return publishOne(conn, key, valueString);
}
exports.updatePartialOne = updatePartialOne;
function updateOnDelete(conn, key, old_val) {
    return publishOne(conn, key, stringifyChanges({
        old_val,
        new_val: null,
    }));
}
exports.updateOnDelete = updateOnDelete;
function updateOne(conn, key, value) {
    return publishOne(conn, key, stringifyChanges({
        old_val: {},
        new_val: value,
    }));
}
exports.updateOne = updateOne;
function publishOne(conn, key, message) {
    return new Promise((resolve, reject) => {
        conn.connection.publish(key, message, (error, _published) => {
            if (error) {
                reject(error);
            }
            else {
                resolve();
            }
        });
    });
}
function observe(conn, key) {
    return new Observable((observer) => {
        //TODO: In future version add wait for the connection and ensure only one call of the quit function
        const connection = conn.connection.duplicate();
        const messageCallback = (_channel, message) => {
            const changes = parseChanges(message);
            debug(`observe ${key} got changes`, changes);
            observer.next(changes);
        };
        connection.on('message', messageCallback);
        connection.subscribe(key);
        return () => {
            connection.removeListener('message', messageCallback);
            connection.unsubscribe(key);
            connection.quit();
        };
    });
}
exports.observe = observe;
const DEFAULT_BATCH_SIZE = 10e3;
exports.CURSOR_BEGINNING = '0';
/**
 * This version of getting keys is same as getKeysByPattern but it using SCAN instead of KEYS.
 * It's better to use for larger set of result keys. For shorter set is better to use KEYS instead.
 */
function scanKeysByPattern(conn, keyPattern, options) {
    return new Promise((resolve, reject) => {
        var _a, _b;
        conn.connection.scan((_a = options === null || options === void 0 ? void 0 : options.cursor) !== null && _a !== void 0 ? _a : exports.CURSOR_BEGINNING, 'MATCH', keyPattern, 'COUNT', ((_b = options === null || options === void 0 ? void 0 : options.batchSize) !== null && _b !== void 0 ? _b : DEFAULT_BATCH_SIZE).toString(), (error, [nextCursor, keys]) => {
            var _a;
            if (error) {
                reject(error);
            }
            else if (nextCursor === exports.CURSOR_BEGINNING || !(options === null || options === void 0 ? void 0 : options.iterateToEnd)) {
                resolve({ keys, nextCursor });
            }
            else {
                (0, wait_1.default)((_a = options === null || options === void 0 ? void 0 : options.iterationDelayMs) !== null && _a !== void 0 ? _a : 0)
                    .then(() => scanKeysByPattern(conn, keyPattern, Object.assign(Object.assign({}, options), { cursor: nextCursor })))
                    .then(({ keys: nextKeys, nextCursor: lastCursor }) => {
                    keys.push(...nextKeys);
                    resolve({ keys, nextCursor: lastCursor });
                })
                    .catch(reject);
            }
        });
    });
}
exports.scanKeysByPattern = scanKeysByPattern;
function addToOrderedSet(conn, key, score, value) {
    return new Promise((resolve, reject) => {
        conn.connection.zadd(key, score, stringifyDocument(value), (error) => (error ? reject(error) : resolve()));
    });
}
exports.addToOrderedSet = addToOrderedSet;
function countOrderedSet(conn, key, minScore, maxScore, minClassification = '[', maxClassification = ']') {
    return new Promise((resolve, reject) => {
        conn.connection.zcount(key, `${minClassification === '[' ? '' : '('}${minScore}`, `${maxClassification === ']' ? '' : '('}${maxScore}`, (error, count) => (error ? reject(error) : resolve(count)));
    });
}
exports.countOrderedSet = countOrderedSet;
function stringifyChanges(changes) {
    return JSON.stringify(Object.assign({ clusterTime: bson_1.Timestamp.fromBits(0, Math.floor(new Date().valueOf() / 1e3)) }, changes));
}
function parseChanges(message) {
    return JSON.parse(message, jsonHelper_1.deserializeJSON);
}
const set = (conn, key, value) => {
    const parsedValue = stringifyDocument(value);
    return new Promise((resolve, reject) => {
        conn.connection.set(key, parsedValue, (error) => {
            if (error) {
                reject(error);
            }
            else {
                resolve();
            }
        });
    });
};
exports.set = set;
const exists = (conn, key) => {
    return new Promise((resolve, reject) => {
        conn.connection.exists(key, (error, value) => {
            if (error) {
                reject(error);
            }
            else {
                resolve(Boolean(value));
            }
        });
    });
};
exports.exists = exists;
const increment = (conn, key) => {
    return new Promise((resolve, reject) => {
        conn.connection.incr(key, (error) => {
            if (error) {
                reject(error);
            }
            else {
                resolve();
            }
        });
    });
};
exports.increment = increment;
//# sourceMappingURL=helper.js.map