"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createDeviceConnectionModel = exports.TTL_SECONDS = exports.deviceDisconnectKey = exports.getDeviceKey = exports.getOrganizationCountsKey = exports.getConnectionKey = exports.connectionKeyPrefix = void 0;
const helper_1 = require("../../helper");
const wait_1 = require("@signageos/lib/dist/Timer/wait");
const Debug = require("debug");
const debug = Debug('@signageos/user-domain-model:Redis:Device:Connection:deviceConnectionModel');
exports.connectionKeyPrefix = 'deviceConnection.uid';
function getConnectionKey(connUid) {
    return `${exports.connectionKeyPrefix}.${connUid}`;
}
exports.getConnectionKey = getConnectionKey;
const organizationCountsKeyPrefix = 'deviceConnection_organizationCounts';
function getOrganizationCountsKey(organizationUid) {
    return `${organizationCountsKeyPrefix}.${organizationUid}`;
}
exports.getOrganizationCountsKey = getOrganizationCountsKey;
function parseOrganizationKey(key) {
    return { organizationUid: key.substring(organizationCountsKeyPrefix.length + 1) };
}
const deviceKeyPrefix = 'deviceConnection_deviceUid';
function getDeviceKey(deviceUid) {
    return `${deviceKeyPrefix}.${deviceUid}`;
}
exports.getDeviceKey = getDeviceKey;
function parseDeviceKey(key) {
    return { deviceUid: key.substring(deviceKeyPrefix.length + 1) };
}
exports.deviceDisconnectKey = 'deviceConnection_disconnect';
exports.TTL_SECONDS = 60;
const createDeviceConnectionModel = (conn, ttlSeconds = exports.TTL_SECONDS) => {
    function computeConnections() {
        var _a, _b;
        return __awaiter(this, void 0, void 0, function* () {
            const organizationsIndex = {};
            const devicesIndex = {};
            let nextConnCursor;
            while (nextConnCursor !== helper_1.CURSOR_BEGINNING) {
                const result = yield (0, helper_1.scanKeysByPattern)(conn, getConnectionKey('*'), { cursor: nextConnCursor });
                nextConnCursor = result.nextCursor;
                for (const key of result.keys) {
                    const deviceConnection = yield (0, helper_1.getOne)(conn, key);
                    if (deviceConnection) {
                        const { organizationUid, deviceUid } = deviceConnection;
                        debug('computeConnections scan result', key, deviceUid, organizationUid);
                        devicesIndex[deviceUid] = (_a = devicesIndex[deviceUid]) !== null && _a !== void 0 ? _a : [];
                        devicesIndex[deviceUid].push(deviceConnection);
                        if (organizationUid) {
                            organizationsIndex[organizationUid] = (_b = organizationsIndex[organizationUid]) !== null && _b !== void 0 ? _b : {};
                            organizationsIndex[organizationUid][deviceUid] = devicesIndex[deviceUid].length;
                        }
                    }
                }
                yield (0, wait_1.default)(100); // give redis time to perform other important queries
            }
            return { organizationsIndex, devicesIndex };
        });
    }
    function writeDeviceConnections(devicesIndex) {
        return __awaiter(this, void 0, void 0, function* () {
            for (const deviceUid in devicesIndex) {
                yield (0, helper_1.deleteOne)(conn, getDeviceKey(deviceUid));
                for (const connection of devicesIndex[deviceUid]) {
                    yield addDeviceConnection(deviceUid, connection);
                }
            }
        });
    }
    function writeOrganizationConnections(organizationsIndex) {
        return __awaiter(this, void 0, void 0, function* () {
            for (const organizationUid in organizationsIndex) {
                yield (0, helper_1.deleteOne)(conn, getOrganizationCountsKey(organizationUid));
                for (const deviceUid in organizationsIndex[organizationUid]) {
                    yield incrementOrganizationConnection(organizationUid, deviceUid, organizationsIndex[organizationUid][deviceUid]);
                }
            }
        });
    }
    function removeExtraDeviceConnections(devicesIndex) {
        return __awaiter(this, void 0, void 0, function* () {
            let nextDevCursor;
            while (nextDevCursor !== helper_1.CURSOR_BEGINNING) {
                const result = yield (0, helper_1.scanKeysByPattern)(conn, getDeviceKey('*'), { cursor: nextDevCursor });
                nextDevCursor = result.nextCursor;
                for (const key of result.keys) {
                    const { deviceUid } = parseDeviceKey(key);
                    if (!(deviceUid in devicesIndex)) {
                        yield (0, helper_1.deleteOne)(conn, key);
                    }
                }
                yield (0, wait_1.default)(100);
            }
        });
    }
    function removeExtraOrganizationConnections(organizationsIndex) {
        return __awaiter(this, void 0, void 0, function* () {
            let nextOrgCursor;
            while (nextOrgCursor !== helper_1.CURSOR_BEGINNING) {
                const result = yield (0, helper_1.scanKeysByPattern)(conn, getOrganizationCountsKey('*'), { cursor: nextOrgCursor });
                nextOrgCursor = result.nextCursor;
                for (const key of result.keys) {
                    const { organizationUid } = parseOrganizationKey(key);
                    if (!(organizationUid in organizationsIndex)) {
                        yield (0, helper_1.deleteOne)(conn, key);
                    }
                }
                yield (0, wait_1.default)(100);
            }
        });
    }
    function addDeviceConnection(deviceUid, connection) {
        return __awaiter(this, void 0, void 0, function* () {
            const deviceConnectionsKey = getDeviceKey(deviceUid);
            yield (0, helper_1.createAndAddToHash)(conn, deviceConnectionsKey, connection.uid, connection);
        });
    }
    function removeDeviceConnection(deviceUid, connection) {
        return __awaiter(this, void 0, void 0, function* () {
            const deviceConnectionsKey = getDeviceKey(deviceUid);
            yield (0, helper_1.deleteOneFromHash)(conn, deviceConnectionsKey, connection.uid);
        });
    }
    function incrementOrganizationConnection(organizationUid, deviceUid, connectionsCount) {
        return __awaiter(this, void 0, void 0, function* () {
            const orgCountsKey = getOrganizationCountsKey(organizationUid);
            yield (0, helper_1.createAndIncrementInHash)(conn, orgCountsKey, deviceUid, connectionsCount);
        });
    }
    return {
        fetchByUid(uid) {
            return __awaiter(this, void 0, void 0, function* () {
                return yield (0, helper_1.getOne)(conn, getConnectionKey(uid));
            });
        },
        fetchListByDevice(device) {
            return __awaiter(this, void 0, void 0, function* () {
                return this.fetchListByDeviceUid(device.uid);
            });
        },
        fetchListByDeviceUid(deviceUid) {
            var _a;
            return __awaiter(this, void 0, void 0, function* () {
                const index = yield this.fetchIndexByDeviceUids([deviceUid]);
                return (_a = index[deviceUid]) !== null && _a !== void 0 ? _a : [];
            });
        },
        fetchIndexByDeviceUids(deviceUids) {
            return __awaiter(this, void 0, void 0, function* () {
                const index = {};
                for (const deviceUid of deviceUids) {
                    const deviceKey = getDeviceKey(deviceUid);
                    const deviceConnections = yield (0, helper_1.getValuesOfHash)(conn, deviceKey);
                    index[deviceUid] = deviceConnections !== null && deviceConnections !== void 0 ? deviceConnections : [];
                }
                return index;
            });
        },
        fetchCountIndexByOrganizationUids(organizationUids) {
            return __awaiter(this, void 0, void 0, function* () {
                const index = {};
                for (const organizationUid of organizationUids) {
                    const orgCountsKey = getOrganizationCountsKey(organizationUid);
                    const countsIndex = yield (0, helper_1.getAllOfHash)(conn, orgCountsKey);
                    index[organizationUid] = {};
                    for (const deviceUid in countsIndex) {
                        index[organizationUid][deviceUid] = parseInt(countsIndex[deviceUid]);
                    }
                }
                return index;
            });
        },
        updateIndexes() {
            return __awaiter(this, void 0, void 0, function* () {
                debug('updateIndexes started');
                const { organizationsIndex, devicesIndex } = yield computeConnections();
                debug('updateIndexes computed', organizationsIndex, devicesIndex);
                yield writeOrganizationConnections(organizationsIndex);
                debug('updateIndexes organization wrote');
                yield writeDeviceConnections(devicesIndex);
                debug('updateIndexes device wrote');
                yield removeExtraOrganizationConnections(organizationsIndex);
                debug('updateIndexes organization removed');
                yield removeExtraDeviceConnections(devicesIndex);
                debug('updateIndexes device removed');
            });
        },
        create(connectionUid, device, serverInstanceUid, createdAt, socketName) {
            return __awaiter(this, void 0, void 0, function* () {
                const connectionKey = getConnectionKey(connectionUid);
                const deviceConnection = {
                    uid: connectionUid,
                    deviceUid: device.uid,
                    serverInstanceUid,
                    socketName,
                    createdAt,
                    organizationUid: device.organizationUid ? device.organizationUid : undefined,
                };
                // create connection itself
                yield (0, helper_1.createOneWithExpiration)(conn, connectionKey, deviceConnection, ttlSeconds);
                // update device index
                yield addDeviceConnection(device.uid, deviceConnection);
                // update organization index
                if (device.organizationUid) {
                    yield incrementOrganizationConnection(device.organizationUid, device.uid, 1);
                }
            });
        },
        remove(record) {
            return __awaiter(this, void 0, void 0, function* () {
                // delete connection itself
                yield (0, helper_1.deleteOne)(conn, getConnectionKey(record.uid));
                // update device index
                yield removeDeviceConnection(record.deviceUid, record);
                // update organization index
                if (record.organizationUid) {
                    yield incrementOrganizationConnection(record.organizationUid, record.deviceUid, -1);
                }
            });
        },
        updateDeviceOrganization(device, organizationUid) {
            return __awaiter(this, void 0, void 0, function* () {
                const dcs = yield this.fetchListByDevice(device);
                const deviceWithOrg = Object.assign(Object.assign({}, device), { organizationUid });
                for (const dc of dcs) {
                    yield this.remove(dc);
                    yield this.create(dc.uid, deviceWithOrg, dc.serverInstanceUid, dc.createdAt, dc.socketName);
                }
            });
        },
        disconnect(duid, socketName, excludeConnectionUid) {
            return __awaiter(this, void 0, void 0, function* () {
                const disconnection = {
                    duid,
                    socketName,
                    excludeConnectionUid,
                };
                yield (0, helper_1.updateOne)(conn, exports.deviceDisconnectKey, disconnection);
            });
        },
        observeDisconnections() {
            return (0, helper_1.observe)(conn, exports.deviceDisconnectKey)
                .filter((changes) => 'new_val' in changes && changes.new_val !== null)
                .map((changes) => changes.new_val);
        },
    };
};
exports.createDeviceConnectionModel = createDeviceConnectionModel;
//# sourceMappingURL=deviceConnectionModel.js.map