"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createPolicyLogModel = exports.preparePolicyLogTable = void 0;
const _ = require("lodash");
const escape_1 = require("@signageos/lib/dist/MongoDB/escape");
const collections_1 = require("../Lib/collections");
const lodash_1 = require("lodash");
const policyLogCollection = (conn) => conn.connection.collection(collections_1.Collection.PolicyLog);
function createRow(uid, createdAt) {
    return {
        uid,
        name: '',
        note: '',
        organizationUid: '',
        items: [],
        archived: false,
        createdAt,
        updatedAt: createdAt,
        originator: {},
    };
}
function updateRowFields(row, fields) {
    return Object.assign(Object.assign({}, row), fields);
}
function omitMongoId(row) {
    return _.omit(row, '_id');
}
const getStatusFilter = (statusFilter) => {
    return [
        (statusFilter === null || statusFilter === void 0 ? void 0 : statusFilter.includeActive) !== undefined && statusFilter.includeActive ? { archived: false, deviceCount: { $gt: 0 } } : {},
        (statusFilter === null || statusFilter === void 0 ? void 0 : statusFilter.includeInactive) !== undefined && statusFilter.includeInactive ? { archived: false, deviceCount: { $eq: 0 } } : {},
        (statusFilter === null || statusFilter === void 0 ? void 0 : statusFilter.includeArchived) !== undefined && statusFilter.includeArchived ? { archived: true } : {},
    ].filter((obj) => Object.keys(obj).length !== 0);
};
const preparePolicyLogTable = (conn) => __awaiter(void 0, void 0, void 0, function* () {
    yield policyLogCollection(conn).createIndex({ uid: 1 }, { name: 'uid', background: true });
    yield policyLogCollection(conn).createIndex({ organizationUid: 1 }, { name: 'organizationUid', background: true });
    yield policyLogCollection(conn).createIndex({ updatedAt: -1 }, { name: 'updatedAt', background: true });
    yield policyLogCollection(conn).createIndex({ uid: 1, updatedAt: -1 }, { name: 'uid_updatedAt', background: true });
    yield policyLogCollection(conn).createIndex({ uid: 1, updatedAt: -1, archived: 1 }, { name: 'uid_updatedAt_archived', background: true });
    yield policyLogCollection(conn).createIndex({ name: 1 }, { name: 'name', background: true });
    yield policyLogCollection(conn).createIndex({ organizationUid: 1, uid: 1, updatedAt: 1 }, { name: 'organizationUid_uid_updatedAt', background: true });
});
exports.preparePolicyLogTable = preparePolicyLogTable;
const createPolicyLogModel = (conn) => ({
    fetchLatestByUid(uid) {
        return policyLogCollection(conn).findOne(
        // Use of uid_updatedAt index
        { uid }, {
            sort: { uid: 1, updatedAt: -1 },
            // End using index
            session: conn.session,
        });
    },
    fetchLogByUid(uid, offset, limit) {
        let pipeline = [
            // Use of uid_updatedAt index
            { $match: { uid } },
            { $sort: { uid: 1, updatedAt: -1 } },
            // End using index
        ];
        const options = { session: conn.session };
        if (offset) {
            pipeline = [...pipeline, { $skip: offset }];
        }
        if (limit) {
            pipeline = [...pipeline, { $limit: limit }];
        }
        return policyLogCollection(conn).aggregate(pipeline, options).toArray();
    },
    fetchLatestByUids(uids) {
        let pipeline = [
            { $match: { uid: { $in: uids } } },
            { $sort: { uid: 1, updatedAt: -1 } },
            {
                $group: {
                    _id: '$uid',
                    latest: { $first: '$$ROOT' },
                },
            },
            { $replaceRoot: { newRoot: '$latest' } },
        ];
        const options = { session: conn.session };
        return policyLogCollection(conn).aggregate(pipeline, options).toArray();
    },
    fetchReferenceListByDevice(device) {
        if (!device.organizationUid) {
            return Promise.resolve([]);
        }
        return policyLogCollection(conn)
            .aggregate([
            // Use of organizationUid_uid_updatedAt index
            { $match: { organizationUid: device.organizationUid } },
            { $sort: { organizationUid: 1, uid: 1, updatedAt: 1 } },
            // End using index
            { $group: { _id: '$uid', latest: { $last: '$$ROOT' } } },
            { $match: { 'latest.archived': false } },
            { $project: { _id: 0, uid: '$latest.uid', name: '$latest.name', organizationUid: '$latest.organizationUid' } },
            { $sort: { name: 1 } },
        ])
            .toArray();
    },
    fetchReferenceListByOrganizations(organizations) {
        const organizationUids = organizations.map((organization) => organization.uid);
        return policyLogCollection(conn)
            .aggregate([
            // Use of organizationUid_uid_updatedAt index
            { $match: { organizationUid: { $in: organizationUids } } },
            { $sort: { organizationUid: 1, uid: 1, updatedAt: 1 } },
            // End using index
            { $group: { _id: '$uid', latest: { $last: '$$ROOT' } } },
            { $match: { 'latest.archived': false } },
            { $project: { _id: 0, uid: '$latest.uid', name: '$latest.name', organizationUid: '$latest.organizationUid' } },
            { $sort: { name: 1 } },
        ])
            .toArray();
    },
    fetchLatestListByOrganizations(organizations, archived, inputSorter, limit, offset, name, createdUntil, statusFilter) {
        const organizationUidList = organizations.map((organization) => organization.uid);
        const options = { session: conn.session };
        const sorter = inputSorter ? (inputSorter instanceof Array ? inputSorter : [inputSorter]) : undefined;
        let pipeline = [
            // Use of organizationUid_uid_updatedAt index
            { $match: { organizationUid: { $in: organizationUidList } } },
            { $sort: { organizationUid: 1, uid: 1, updatedAt: 1 } },
            // End using index
            { $group: { _id: '$uid', latest: { $last: '$$ROOT' } } },
            { $sort: { _id: 1 } },
            { $replaceRoot: { newRoot: '$latest' } },
        ];
        if (name) {
            pipeline = [...pipeline, { $match: { name: { $regex: (0, lodash_1.escapeRegExp)(name), $options: 'i' } } }];
        }
        if (createdUntil) {
            pipeline = [...pipeline, { $match: { createdAt: { $lt: createdUntil } } }];
        }
        if (typeof statusFilter === 'undefined') {
            pipeline = [...pipeline, { $match: { archived: !!archived } }];
        }
        if ((sorter && sorter.some((s) => s.field === 'deviceCount')) || statusFilter) {
            pipeline = [
                ...pipeline,
                {
                    $lookup: {
                        from: 'device',
                        foreignField: 'policies.uid',
                        localField: 'uid',
                        as: 'device',
                    },
                },
                {
                    $addFields: {
                        deviceCount: { $size: '$device' },
                    },
                },
            ];
        }
        if (statusFilter) {
            const filterStatus = getStatusFilter(statusFilter);
            if (filterStatus.length > 0) {
                pipeline = [...pipeline, { $match: { $or: filterStatus } }];
            }
        }
        if (sorter) {
            const hasDeviceCountSorter = sorter.some((s) => s.field === 'deviceCount');
            const hasOrganizationTitleSorter = sorter.some((s) => s.field === 'organizationTitle');
            pipeline = [
                ...pipeline,
                ...(hasOrganizationTitleSorter
                    ? [
                        {
                            $lookup: {
                                from: 'organization',
                                foreignField: 'uid',
                                localField: 'organizationUid',
                                as: 'organization',
                            },
                        },
                    ]
                    : []),
                {
                    $sort: sorter.reduce((acc, sortOption) => {
                        const sortDirection = sortOption.direction === 'descending' ? -1 : 1;
                        switch (sortOption.field) {
                            case 'name':
                            case 'createdAt':
                            case 'updatedAt':
                                acc[sortOption.field] = sortDirection;
                                break;
                            case 'deviceCount':
                                acc.deviceCount = sortDirection;
                                break;
                            case 'organizationTitle':
                                acc['organization.title'] = sortDirection;
                                break;
                            default:
                                break;
                        }
                        return acc;
                    }, {}),
                },
                ...(hasDeviceCountSorter || hasOrganizationTitleSorter
                    ? [
                        {
                            $project: Object.assign(Object.assign({}, (hasDeviceCountSorter ? { device: 0, deviceCount: 0 } : {})), (hasOrganizationTitleSorter ? { organization: 0 } : {})),
                        },
                    ]
                    : []),
            ];
        }
        if (offset) {
            pipeline = [...pipeline, { $skip: offset }];
        }
        if (limit) {
            pipeline = [...pipeline, { $limit: limit }];
        }
        return policyLogCollection(conn).aggregate(pipeline, options).toArray();
    },
    fetchLatestByOrganizationAndUid(organization, uid) {
        const organizationUid = organization.uid;
        return policyLogCollection(conn).findOne(
        // Use of organizationUid_uid_updatedAt index
        { organizationUid, uid }, {
            sort: { organizationUid: 1, uid: 1, updatedAt: -1 },
            // End using index
            session: conn.session,
        });
    },
    countByUid(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            const count = yield policyLogCollection(conn).countDocuments(
            // Use of uid index
            { uid }, 
            // End using index
            { session: conn.session });
            return count;
        });
    },
    countLatestByOrganization(organizations, archived, name, statusFilter) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            const organizationUidList = organizations.map((organization) => organization.uid);
            const options = { session: conn.session };
            const filterStatus = getStatusFilter(statusFilter);
            let pipeline = [
                // Use of organizationUid_uid_updatedAt index
                {
                    $match: {
                        $and: [{ organizationUid: { $in: organizationUidList } }].concat(name ? { name: { $regex: (0, lodash_1.escapeRegExp)(name), $options: 'gi' } } : []),
                    },
                },
                { $sort: { organizationUid: 1, uid: 1, updatedAt: 1 } },
                // End using index
                { $group: { _id: '$uid', latest: { $last: '$$ROOT' } } },
                { $replaceRoot: { newRoot: '$latest' } },
            ];
            if (typeof statusFilter === 'undefined') {
                pipeline = [...pipeline, { $match: { archived: !!archived } }];
            }
            if (filterStatus.length > 0) {
                pipeline = [
                    ...pipeline,
                    {
                        $lookup: {
                            from: 'device',
                            foreignField: 'policies.uid',
                            localField: 'uid',
                            as: 'device',
                        },
                    },
                    {
                        $addFields: {
                            deviceCount: { $size: '$device' },
                        },
                    },
                    {
                        $match: {
                            $or: filterStatus,
                        },
                    },
                ];
            }
            pipeline = [...pipeline, { $count: 'count' }];
            const [result] = yield policyLogCollection(conn).aggregate(pipeline, options).toArray();
            return (_a = result === null || result === void 0 ? void 0 : result.count) !== null && _a !== void 0 ? _a : 0;
        });
    },
    create(uid, name, organizationUid, createdAt, originator) {
        return __awaiter(this, void 0, void 0, function* () {
            const newRow = updateRowFields(createRow(uid, createdAt), {
                name,
                note: 'New policy created',
                organizationUid,
                originator,
            });
            yield policyLogCollection(conn).insertOne(newRow, { session: conn.session });
        });
    },
    clone(policyLogRow, uid, name, organizationUid, createdAt, originator) {
        return __awaiter(this, void 0, void 0, function* () {
            const clonedRow = updateRowFields(createRow(uid, createdAt), {
                name,
                note: `Policy cloned from ${policyLogRow.name}`,
                organizationUid,
                items: (0, escape_1.escapeObject)(policyLogRow.items),
                originator,
            });
            yield policyLogCollection(conn).insertOne(clonedRow, { session: conn.session });
        });
    },
    updateItems(policyLogRow, items, updatedAt, originator, note) {
        return __awaiter(this, void 0, void 0, function* () {
            const updatedRow = updateRowFields(omitMongoId(policyLogRow), Object.assign(Object.assign({}, (note ? { note } : {})), { items: (0, escape_1.escapeObject)(items), updatedAt,
                originator }));
            yield policyLogCollection(conn).insertOne(updatedRow, { session: conn.session });
        });
    },
    updateName(policyLogRow, name, updatedAt, originator) {
        return __awaiter(this, void 0, void 0, function* () {
            const updatedRow = updateRowFields(omitMongoId(policyLogRow), {
                name,
                note: 'Name updated',
                updatedAt,
                originator,
            });
            yield policyLogCollection(conn).insertOne(updatedRow, { session: conn.session });
        });
    },
    archive(policyLogRow, updatedAt, originator) {
        return __awaiter(this, void 0, void 0, function* () {
            const archivedRow = updateRowFields(omitMongoId(policyLogRow), {
                archived: true,
                note: 'Archived',
                updatedAt,
                originator,
            });
            yield policyLogCollection(conn).insertOne(archivedRow, { session: conn.session });
        });
    },
    unarchive(policyLogRow, updatedAt, originator) {
        return __awaiter(this, void 0, void 0, function* () {
            const unarchivedRow = updateRowFields(omitMongoId(policyLogRow), {
                archived: false,
                note: 'Unarchived',
                updatedAt,
                originator,
            });
            yield policyLogCollection(conn).insertOne(unarchivedRow, { session: conn.session });
        });
    },
    remove(policyLogRow) {
        return __awaiter(this, void 0, void 0, function* () {
            yield policyLogCollection(conn).deleteMany({ uid: policyLogRow.uid });
        });
    },
});
exports.createPolicyLogModel = createPolicyLogModel;
//# sourceMappingURL=policyLogModel.js.map