"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createStatisticsDatesFilter = exports.removeObsoleteDevicePlans = exports.createOrganizationModel = exports.prepareOrganizationTable = exports.organizationCollection = void 0;
const lodash_1 = require("lodash");
const Billing_1 = require("@signageos/common-types/dist/Billing/Billing");
const object_1 = require("@signageos/lib/dist/Utils/object");
const organizationModel_1 = require("../../Schema/Organization/organizationModel");
const collections_1 = require("../Lib/collections");
const ICursor_1 = require("../../Cursor/ICursor");
const organizationCollection = (conn) => conn.connection.collection(collections_1.Collection.Organization);
exports.organizationCollection = organizationCollection;
const prepareOrganizationTable = (conn) => __awaiter(void 0, void 0, void 0, function* () {
    yield (0, exports.organizationCollection)(conn).createIndex('uid', { name: 'uid', unique: true, background: true });
    yield (0, exports.organizationCollection)(conn).createIndex('isCompany', { name: 'isCompany', background: true });
    yield (0, exports.organizationCollection)(conn).createIndex('productionSince', { name: 'productionSince', background: true });
    yield (0, exports.organizationCollection)(conn).createIndex('subscriptionType', { name: 'subscriptionType', background: true });
    yield (0, exports.organizationCollection)(conn).createIndex('billingPlan', { name: 'billingPlan', background: true });
    yield (0, exports.organizationCollection)(conn).createIndex('createdAt', { name: 'createdAt', background: true });
    yield (0, exports.organizationCollection)(conn).createIndex('oauthClientId', { name: 'oauthClientId', background: true });
    yield (0, exports.organizationCollection)(conn).createIndex('accountIds', { name: 'accountIds', background: true });
    yield (0, exports.organizationCollection)(conn).createIndex('emailDomains', { name: 'emailDomains', background: true, unique: true, sparse: true });
});
exports.prepareOrganizationTable = prepareOrganizationTable;
const getSorter = (sorter) => {
    const sortDirection = sorter.direction === 'descending' ? organizationModel_1.OrderDirection.DESC : organizationModel_1.OrderDirection.ASC;
    if (sorter.field === 'title') {
        return { title: sortDirection };
    }
    else if (sorter.field === 'createdAt') {
        return { createdAt: sortDirection };
    }
    else if (sorter.field === 'uid') {
        return { uid: sortDirection };
    }
    else if (sorter.field === 'subscriptionType') {
        return { subscriptionType: sortDirection };
    }
    else if (sorter.field === 'company') {
        return { 'company.title': sortDirection };
    }
    return { title: organizationModel_1.OrderDirection.ASC };
};
const createPipelineFromFilter = (options) => {
    const pipeline = [];
    const { search, sorter, filter, pagination } = options;
    pipeline.push({
        $match: Object.assign(search
            ? {
                $or: [
                    {
                        name: { $regex: (0, lodash_1.escapeRegExp)(search), $options: 'i' },
                    },
                    {
                        title: { $regex: (0, lodash_1.escapeRegExp)(search), $options: 'i' },
                    },
                    {
                        uid: { $regex: (0, lodash_1.escapeRegExp)(search), $options: 'i' },
                    },
                ],
            }
            : {}, (filter === null || filter === void 0 ? void 0 : filter.isCompany) ? { isCompany: filter.isCompany } : {}, (filter === null || filter === void 0 ? void 0 : filter.accountId) ? { accountIds: filter.accountId } : {}, (filter === null || filter === void 0 ? void 0 : filter.parentUid) ? { parentUid: filter.parentUid } : {}, (filter === null || filter === void 0 ? void 0 : filter.companyNetworkUids) ? { companyNetworkUids: { $in: filter.companyNetworkUids } } : {}, (filter === null || filter === void 0 ? void 0 : filter.billingPlan) ? { billingPlan: filter.billingPlan } : {}, (filter === null || filter === void 0 ? void 0 : filter.trialPeriodEndedAtRange)
            ? {
                $and: [
                    {
                        trialPeriodEndedAt: { $gte: filter.trialPeriodEndedAtRange.from },
                    },
                    {
                        trialPeriodEndedAt: { $lte: filter.trialPeriodEndedAtRange.to },
                    },
                ],
            }
            : {}),
    });
    pipeline.push({
        $sort: sorter && Object.keys(sorter).length > 0
            ? getSorter(sorter)
            : {
                createdAt: organizationModel_1.OrderDirection.DESC,
                title: organizationModel_1.OrderDirection.ASC,
            },
    });
    if (pagination === null || pagination === void 0 ? void 0 : pagination.offset) {
        pipeline.push({
            $skip: pagination.offset,
        });
    }
    if (pagination === null || pagination === void 0 ? void 0 : pagination.limit) {
        pipeline.push({
            $limit: pagination.limit,
        });
    }
    return pipeline;
};
const createOrganizationModel = (conn) => {
    const ensureOrganizationsList = (list) => {
        const ensured = [];
        for (let item of list) {
            ensured.push(ensureOrganization(item));
        }
        return ensured;
    };
    /** ensureOrganization is responsible that IOrganization is semantically valid.
     *   - Checks mandatory fields exist
     *   - Fills default values to fields that are inconsistent in the storage
     */
    const ensureOrganization = (o) => {
        if (o === null) {
            return null;
        }
        if (!o.uid) {
            throw new Error("Invalid Organization document. Field 'uid' is missing.");
        }
        if (typeof o.isCompany === 'undefined') {
            // whenever the parentUid is set, the organization IS NOT a company
            if (o.parentUid) {
                o.isCompany = false;
            }
            else {
                o.isCompany = true;
            }
            console.warn(`Organization ${o.uid} ambiguous isCompany. Should be marked with isCompany=${o.isCompany}`);
        }
        if (o.isCompany) {
            if (typeof o.billingPlan === 'undefined') {
                o.billingPlan = Billing_1.BillingPlan.Open;
            }
        }
        else {
            if (typeof o.subscriptionType === 'undefined') {
                o.subscriptionType = organizationModel_1.SubscriptionType.Open;
            }
        }
        if (typeof o.accountIds === 'undefined') {
            o.accountIds = [];
        }
        return o;
    };
    return {
        createCompany(uid, name, title, oauthClientId, oauthClientSecret, createdAt, billingPlan, iconUrl) {
            return __awaiter(this, void 0, void 0, function* () {
                const newDoc = {
                    uid,
                    name,
                    title,
                    createdAt,
                    oauthClientId,
                    oauthClientSecret,
                    billingPlan,
                    isCompany: true,
                    accountIds: [],
                    iconUrl,
                };
                yield (0, exports.organizationCollection)(conn).insertOne(newDoc, { session: conn.session });
            });
        },
        updateCompany({ organization, title, emailDomains, defaultAccountRole, iconUrl }) {
            return __awaiter(this, void 0, void 0, function* () {
                const updatedFields = (0, object_1.getObjectWithOmittedValuesBy)({ title, emailDomains, defaultAccountRole, iconUrl }, undefined);
                const isEmailDomainsEmpty = Array.isArray(updatedFields.emailDomains) && (updatedFields === null || updatedFields === void 0 ? void 0 : updatedFields.emailDomains.length) === 0;
                const getUpdate = () => {
                    var _a;
                    let set = {};
                    let unset = {};
                    let result = {};
                    // unset/set emailDomains
                    if (isEmailDomainsEmpty) {
                        unset = Object.assign(Object.assign({}, unset), { emailDomains: '' });
                    }
                    else if (((_a = updatedFields.emailDomains) !== null && _a !== void 0 ? _a : []).length > 0) {
                        set = {
                            emailDomains: updatedFields.emailDomains,
                        };
                    }
                    // unset/set defaultAccountRole
                    if (updatedFields.defaultAccountRole === null) {
                        unset = Object.assign(Object.assign({}, unset), { defaultAccountRole: '' });
                    }
                    else if (updatedFields.defaultAccountRole) {
                        set = {
                            defaultAccountRole: updatedFields.defaultAccountRole,
                        };
                    }
                    set = Object.assign(Object.assign({}, (0, lodash_1.omit)(updatedFields, ['emailDomains', 'defaultAccountRole'])), set);
                    if (Object.keys(set).length > 0) {
                        result = {
                            $set: set,
                        };
                    }
                    if (Object.keys(unset).length > 0) {
                        result = Object.assign(Object.assign({}, result), { $unset: unset });
                    }
                    return result;
                };
                const update = getUpdate();
                if (Object.keys(update).length > 0) {
                    // prettier-ignore
                    yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, update, { session: conn.session });
                }
            });
        },
        createOrganization(uid, name, title, oauthClientId, oauthClientSecret, createdAt, subscriptionType, iconUrl) {
            return __awaiter(this, void 0, void 0, function* () {
                (0, organizationModel_1.validateSubscriptionType)(subscriptionType);
                const newDoc = {
                    uid,
                    name,
                    title,
                    createdAt,
                    oauthClientId,
                    oauthClientSecret,
                    subscriptionType,
                    isCompany: false,
                    accountIds: [],
                    iconUrl,
                };
                yield (0, exports.organizationCollection)(conn).insertOne(newDoc, { session: conn.session });
            });
        },
        fetchByUid(uid) {
            return __awaiter(this, void 0, void 0, function* () {
                return ensureOrganization(yield (0, exports.organizationCollection)(conn).findOne({ uid }, { session: conn.session }));
            });
        },
        fetchByOAuthClientId(oauthClientId) {
            return __awaiter(this, void 0, void 0, function* () {
                return ensureOrganization(yield (0, exports.organizationCollection)(conn).findOne({ oauthClientId }, { session: conn.session }));
            });
        },
        fetchByName(name) {
            return __awaiter(this, void 0, void 0, function* () {
                return ensureOrganization(yield (0, exports.organizationCollection)(conn).findOne({ name }, { session: conn.session }));
            });
        },
        fetchCompaniesByEmailDomains({ emailDomains }) {
            return __awaiter(this, void 0, void 0, function* () {
                const list = yield (0, exports.organizationCollection)(conn)
                    .aggregate([
                    {
                        $match: { isCompany: true, emailDomains: { $in: emailDomains } },
                    },
                    { $sort: getSorter({ field: 'title', direction: 'descending' }) },
                ])
                    .toArray();
                return ensureOrganizationsList(list);
            });
        },
        fetchCompanyByEmailDomain({ emailDomain }) {
            return __awaiter(this, void 0, void 0, function* () {
                return ensureOrganization(yield (0, exports.organizationCollection)(conn).findOne({ isCompany: true, emailDomains: [emailDomain] }, { session: conn.session }));
            });
        },
        fetchListByUids(uids) {
            return __awaiter(this, void 0, void 0, function* () {
                const list = yield (0, exports.organizationCollection)(conn)
                    .find({ uid: { $in: uids } }, { session: conn.session })
                    .toArray();
                return ensureOrganizationsList(list);
            });
        },
        fetchAllList() {
            return __awaiter(this, void 0, void 0, function* () {
                const list = yield (0, exports.organizationCollection)(conn).find({}, { session: conn.session }).toArray();
                return ensureOrganizationsList(list);
            });
        },
        getIterator() {
            return __awaiter(this, void 0, void 0, function* () {
                const organizationCursor = (0, exports.organizationCollection)(conn).aggregate([], { session: conn.session });
                return (0, ICursor_1.cursorIterator)(organizationCursor);
            });
        },
        getAllOrganizationsWithPoliciesAndDevicesIterator() {
            return __awaiter(this, void 0, void 0, function* () {
                const organizationCursor = (0, exports.organizationCollection)(conn).aggregate([
                    {
                        $match: { isCompany: false },
                    },
                    {
                        $lookup: {
                            from: 'policy',
                            let: { orgUid: '$uid' },
                            pipeline: [
                                { $match: { $expr: { $eq: ['$$orgUid', '$organizationUid'] } } },
                                {
                                    $lookup: {
                                        from: 'device',
                                        localField: 'uid',
                                        foreignField: 'policies.uid',
                                        as: 'devices',
                                    },
                                },
                            ],
                            as: 'orgPolicies',
                        },
                    },
                ], { session: conn.session });
                return (0, ICursor_1.cursorIterator)(organizationCursor);
            });
        },
        fetchAllOrganizationsList() {
            return __awaiter(this, void 0, void 0, function* () {
                const list = yield (0, exports.organizationCollection)(conn).find({ isCompany: false }, { session: conn.session }).toArray();
                return ensureOrganizationsList(list);
            });
        },
        fetchTotalCountByAccount(account, filter) {
            var _a, _b;
            return __awaiter(this, void 0, void 0, function* () {
                const result = yield (0, exports.organizationCollection)(conn)
                    .aggregate([
                    {
                        $facet: {
                            organizationsTotalCount: [
                                {
                                    $match: Object.assign(Object.assign({ accountIds: account.id, isCompany: false }, ((filter === null || filter === void 0 ? void 0 : filter.name) ? { title: { $regex: (0, lodash_1.escapeRegExp)(filter.name), $options: 'i' } } : {})), ((filter === null || filter === void 0 ? void 0 : filter.parentUid) ? { parentUid: filter.parentUid } : {})),
                                },
                                { $count: 'count' },
                            ],
                        },
                    },
                ])
                    .toArray();
                return (_b = (_a = result[0].organizationsTotalCount) === null || _a === void 0 ? void 0 : _a[0]) === null || _b === void 0 ? void 0 : _b.count;
            });
        },
        fetchSortedListByFilterAndAccount(options) {
            var _a, _b;
            return __awaiter(this, void 0, void 0, function* () {
                const { account, sorter, filter, pagination } = options;
                const result = yield (0, exports.organizationCollection)(conn)
                    .aggregate([
                    {
                        $facet: {
                            organizations: [
                                {
                                    $match: Object.assign(Object.assign({ isCompany: false, accountIds: account.id }, ((filter === null || filter === void 0 ? void 0 : filter.name) ? { title: { $regex: (0, lodash_1.escapeRegExp)(filter.name), $options: 'i' } } : {})), ((filter === null || filter === void 0 ? void 0 : filter.parentUid) ? { parentUid: filter.parentUid } : {})),
                                },
                                {
                                    $lookup: {
                                        from: 'organization',
                                        localField: 'parentUid',
                                        foreignField: 'uid',
                                        as: 'company',
                                    },
                                },
                            ].concat(sorter && Object.keys(sorter).length > 0 ? { $sort: getSorter(sorter) } : [], pagination && Object.keys(pagination).length > 0 ? [{ $skip: pagination.offset }, { $limit: pagination.limit }] : []),
                            companies: [{ $match: { accountIds: account.id, isCompany: true } }],
                            organizationsTotalCount: [
                                {
                                    $match: Object.assign(Object.assign({ isCompany: false, accountIds: account.id }, ((filter === null || filter === void 0 ? void 0 : filter.name) ? { title: { $regex: (0, lodash_1.escapeRegExp)(filter.name), $options: 'i' } } : {})), ((filter === null || filter === void 0 ? void 0 : filter.parentUid) ? { parentUid: filter.parentUid } : {})),
                                },
                                {
                                    $count: 'count',
                                },
                            ],
                        },
                    },
                ])
                    .toArray();
                return {
                    companies: result[0].companies,
                    organizations: result[0].organizations,
                    organizationsTotalCount: ((_b = (_a = result[0].organizationsTotalCount) === null || _a === void 0 ? void 0 : _a[0]) === null || _b === void 0 ? void 0 : _b.count) || 0,
                };
            });
        },
        fetchAllCompaniesList() {
            return __awaiter(this, void 0, void 0, function* () {
                const list = yield (0, exports.organizationCollection)(conn).find({ isCompany: true }, { session: conn.session }).toArray();
                return ensureOrganizationsList(list);
            });
        },
        fetchAllByFilter(options) {
            return __awaiter(this, void 0, void 0, function* () {
                const pipeline = createPipelineFromFilter(options);
                return yield (0, exports.organizationCollection)(conn).aggregate(pipeline).toArray();
            });
        },
        countAllByFilter(options) {
            var _a;
            return __awaiter(this, void 0, void 0, function* () {
                let pipeline = createPipelineFromFilter(options);
                pipeline = [...pipeline, { $count: 'count' }];
                const [result] = yield (0, exports.organizationCollection)(conn).aggregate(pipeline, { session: conn.session }).toArray();
                return (_a = result === null || result === void 0 ? void 0 : result.count) !== null && _a !== void 0 ? _a : 0;
            });
        },
        fetchListByAccount(account) {
            return __awaiter(this, void 0, void 0, function* () {
                const list = yield (0, exports.organizationCollection)(conn).find({ accountIds: account.id }, { session: conn.session }).toArray();
                return ensureOrganizationsList(list);
            });
        },
        fetchListByPropertyFilter(propertyFilter) {
            return __awaiter(this, void 0, void 0, function* () {
                const filter = convertPropertyFilterToMongoFilter(propertyFilter);
                const list = yield (0, exports.organizationCollection)(conn).find(filter, { session: conn.session }).toArray();
                return ensureOrganizationsList(list);
            });
        },
        fetchListByAccountAndFilter(account, propertyFilter) {
            return __awaiter(this, void 0, void 0, function* () {
                propertyFilter.accountId = account.id;
                const list = yield this.fetchListByPropertyFilter(propertyFilter);
                return ensureOrganizationsList(list);
            });
        },
        updateTitle(org, title) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: org.uid }, { $set: { title } }, { session: conn.session });
            });
        },
        updateIconUrl(org, iconUrl) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: org.uid }, { $set: { iconUrl } }, { session: conn.session });
            });
        },
        updateDefaultLocationMarker(org, defaultLocationMarker) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: org.uid }, { $set: { defaultLocationMarker } }, { session: conn.session });
            });
        },
        updateVPNUrl(org, vpnServerNameTag) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: org.uid }, { $set: { vpnServerNameTag } }, { session: conn.session });
            });
        },
        deleteVPNUrl(org) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: org.uid }, { $unset: { vpnServerNameTag: '' } }, { session: conn.session });
            });
        },
        updateBillingPlan(org, billingPlan) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: org.uid }, { $set: { billingPlan } }, { session: conn.session });
            });
        },
        updateBillingInfo(org, billingInfo) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: org.uid }, { $set: { billingInfo } }, { session: conn.session });
            });
        },
        changeSubscriptionType(org, subscriptionType) {
            return __awaiter(this, void 0, void 0, function* () {
                (0, organizationModel_1.validateSubscriptionType)(subscriptionType);
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: org.uid }, { $set: { subscriptionType } }, { session: conn.session });
            });
        },
        setDefaultSubscriptionType(company, defaultSubscriptionType) {
            return __awaiter(this, void 0, void 0, function* () {
                (0, organizationModel_1.validateSubscriptionType)(defaultSubscriptionType);
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: company.uid }, { $set: { defaultSubscriptionType } }, { session: conn.session });
            });
        },
        assignAccount(org, account, _assignedAt) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: org.uid }, { $addToSet: { accountIds: account.id } }, { session: conn.session });
            });
        },
        removeAccount(org, account) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: org.uid }, { $pull: { accountIds: account.id } }, { session: conn.session });
            });
        },
        remove(org) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).deleteOne({ uid: org.uid }, { session: conn.session });
            });
        },
        //
        // Organizations tree methods
        //
        isParent(org) {
            return !org.parentUid;
        },
        isChild(org) {
            return !!org.parentUid;
        },
        getParent(org) {
            return __awaiter(this, void 0, void 0, function* () {
                const current = yield this.fetchByUid(org.uid);
                if (!current || !this.isChild(current)) {
                    throw new Error('Cannot getParent of non childish organization');
                }
                // @ts-ignore: org.parentUid is not null
                return this.fetchByUid(current.parentUid);
            });
        },
        getChildren(org, pagination, sorter) {
            var _a;
            return __awaiter(this, void 0, void 0, function* () {
                const current = yield this.fetchByUid(org.uid);
                if (!current || !this.isParent(current)) {
                    throw new Error('Cannot getParent of non parental organization');
                }
                const pipeline = [
                    {
                        $match: {
                            parentUid: org.uid,
                        },
                    },
                ];
                if (sorter && Object.keys(sorter).length > 0) {
                    pipeline.push({
                        $sort: getSorter(sorter),
                    });
                }
                if (pagination) {
                    if (pagination.offset) {
                        pipeline.push({
                            $skip: (_a = pagination.offset) !== null && _a !== void 0 ? _a : 0,
                        });
                    }
                    if (pagination.limit) {
                        pipeline.push({
                            $limit: pagination.limit,
                        });
                    }
                }
                return yield (0, exports.organizationCollection)(conn).aggregate(pipeline, { session: conn.session }).toArray();
            });
        },
        getChildrenCount(org) {
            return __awaiter(this, void 0, void 0, function* () {
                const current = yield this.fetchByUid(org.uid);
                if (!current || !this.isParent(current)) {
                    throw new Error('Cannot getParent of non parental organization');
                }
                return (0, exports.organizationCollection)(conn).find({ parentUid: current.uid }, { session: conn.session }).count();
            });
        },
        getChildrenWithFilter(org, filter) {
            return __awaiter(this, void 0, void 0, function* () {
                const result = yield (0, exports.organizationCollection)(conn)
                    .aggregate([
                    {
                        $match: Object.assign({ parentUid: org.uid }, (filter.name ? { title: { $regex: (0, lodash_1.escapeRegExp)(filter.name), $options: 'i' } } : {})),
                    },
                ])
                    .toArray();
                return result;
            });
        },
        setParent(child, parent) {
            return __awaiter(this, void 0, void 0, function* () {
                const currentChild = yield this.fetchByUid(child.uid);
                if (!currentChild) {
                    throw new Error('Child does not exist');
                }
                const currentParent = yield this.fetchByUid(parent.uid);
                if (!currentParent) {
                    throw new Error('Parent does not exist');
                }
                // prevent tree with more then 2 layers
                if (this.isChild(currentParent)) {
                    throw new Error('Desired parent organization is already a child.');
                }
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: currentChild.uid }, { $set: { parentUid: currentParent.uid } }, { session: conn.session });
            });
        },
        unsetParent(organization) {
            return __awaiter(this, void 0, void 0, function* () {
                const currentOrg = yield this.fetchByUid(organization.uid);
                if (!currentOrg) {
                    throw new Error('Organization does not exist');
                }
                if (!currentOrg.parentUid) {
                    throw new Error('Organization does not have any parental organization');
                }
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: currentOrg.uid }, { $unset: { parentUid: '' } }, { session: conn.session });
            });
        },
        updateProductionSince(organization, productionSince) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, { $set: { productionSince } }, { session: conn.session });
            });
        },
        deleteProductionSince(organization) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, { $unset: { productionSince: '' } }, { session: conn.session });
            });
        },
        updateMaxDevicesCount(organization, maxDevicesCount) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, { $set: { maxDevicesCount } }, { session: conn.session });
            });
        },
        fetchCompaniesDistributionByBillingPlan(createdSince, createdBefore) {
            return __awaiter(this, void 0, void 0, function* () {
                let filter = createStatisticsDatesFilter(createdSince, createdBefore);
                filter = Object.assign(Object.assign({}, filter), { isCompany: true });
                const aggregationResult = yield (0, exports.organizationCollection)(conn)
                    .aggregate([{ $match: filter }, { $group: { _id: '$billingPlan', total: { $sum: 1 } } }])
                    .toArray();
                const defaultCounts = (0, organizationModel_1.createDefaultDistributionByBillingPlan)();
                return aggregationResult.reduce((accumulator, carry) => {
                    if (carry._id === null) {
                        return accumulator;
                    }
                    return Object.assign(Object.assign({}, accumulator), { [carry._id]: carry.total });
                }, defaultCounts);
            });
        },
        fetchOrganizationsDistributionByDevicePlan(isProduction, createdSince, createdBefore) {
            return __awaiter(this, void 0, void 0, function* () {
                let filter = createStatisticsDatesFilter(createdSince, createdBefore);
                filter = Object.assign(Object.assign({}, filter), { isCompany: false });
                if (isProduction !== undefined) {
                    if (isProduction) {
                        filter = Object.assign(Object.assign({}, filter), { productionSince: { $ne: '', $exists: true } });
                    }
                    else {
                        filter = Object.assign(Object.assign({}, filter), { productionSince: { $exists: false } });
                    }
                }
                const aggregationResult = yield (0, exports.organizationCollection)(conn)
                    .aggregate([{ $match: filter }, { $group: { _id: '$subscriptionType', total: { $sum: 1 } } }])
                    .toArray();
                const defaultCounts = (0, organizationModel_1.createDefaultDistributionByDevicePlan)();
                const counts = aggregationResult.reduce((accumulator, carry) => {
                    if (carry._id === null) {
                        return accumulator;
                    }
                    return Object.assign(Object.assign({}, accumulator), { [carry._id]: carry.total });
                }, defaultCounts);
                return removeObsoleteDevicePlans(counts);
            });
        },
        fetchOrganizationByUidDevicesCount(organizationUid) {
            return __awaiter(this, void 0, void 0, function* () {
                const cursor = (0, exports.organizationCollection)(conn).aggregate([
                    { $match: { uid: organizationUid, isCompany: false } },
                    {
                        $lookup: {
                            from: 'device',
                            localField: 'uid',
                            foreignField: 'organizationUid',
                            as: 'device',
                        },
                    },
                    { $addFields: { deviceCount: { $size: '$device' } } },
                    {
                        $project: {
                            device: 0,
                        },
                    },
                ]);
                return (yield cursor.toArray())[0];
            });
        },
        fetchOrganizationsByCompanyUid(companyUid) {
            return (0, exports.organizationCollection)(conn)
                .aggregate([{ $match: { parentUid: companyUid } }])
                .toArray();
        },
        updateEndOfTrialPeriod(organization, endDate) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, {
                    $set: { trialPeriodEndedAt: endDate },
                }, { session: conn.session });
            });
        },
        clearEndOfTrialPeriod(organization) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, {
                    $unset: { trialPeriodEndedAt: '' },
                }, { session: conn.session });
            });
        },
        setDeviceNameTemplate(organization, template) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, {
                    $set: { deviceNameTemplate: template },
                }, { session: conn.session });
            });
        },
        fetchOrganizationCompanyMap(uids) {
            return __awaiter(this, void 0, void 0, function* () {
                const organizationsWithParents = yield (0, exports.organizationCollection)(conn)
                    .find({
                    uid: {
                        $in: uids,
                    },
                }, {
                    projection: { _id: 0, uid: 1, parentUid: 1 },
                    session: conn.session,
                })
                    .toArray();
                const organizationCompanyMap = new Map();
                for (const organizationWithParent of organizationsWithParents) {
                    organizationCompanyMap.set(organizationWithParent.uid, organizationWithParent.parentUid);
                }
                return organizationCompanyMap;
            });
        },
        assignCompanyNetworkToOrganization({ organization, companyNetwork, }) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, { $push: { companyNetworkUids: companyNetwork.uid } }, { session: conn.session });
            });
        },
        unassignCompanyNetworkFromOrganization({ organization, companyNetwork, }) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, { $pull: { companyNetworkUids: companyNetwork.uid } }, { session: conn.session });
            });
        },
        assignWhiteLabelSettingsToOrganization({ organization, whiteLabelSettings, }) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, { $set: { whiteLabelSettingsUid: whiteLabelSettings.uid } }, { session: conn.session });
            });
        },
        unassignWhiteLabelSettingsFromOrganization(organization) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, { $unset: { whiteLabelSettingsUid: '' } }, { session: conn.session });
            });
        },
        updateOrganizationReporters(organization, reporters) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.organizationCollection)(conn).updateOne({ uid: organization.uid }, { $set: { reporters: reporters } }, { session: conn.session });
            });
        },
    };
};
exports.createOrganizationModel = createOrganizationModel;
function removeObsoleteDevicePlans(counts) {
    // every "Platform" consider as "All"
    if (organizationModel_1.SubscriptionType.Platform in counts) {
        counts[organizationModel_1.SubscriptionType.All] = counts[organizationModel_1.SubscriptionType.All] + counts[organizationModel_1.SubscriptionType.Platform];
        delete counts[organizationModel_1.SubscriptionType.Platform];
    }
    return counts;
}
exports.removeObsoleteDevicePlans = removeObsoleteDevicePlans;
function createStatisticsDatesFilter(createdSince, createdBefore) {
    let filter = {};
    if (createdSince) {
        filter = Object.assign(Object.assign({}, filter), { createdAt: { $gte: createdSince } });
    }
    if (createdBefore) {
        if (createdSince) {
            filter = Object.assign(Object.assign({}, filter), { createdAt: { $gte: createdSince, $lte: createdBefore } });
        }
        else {
            filter = Object.assign(Object.assign({}, filter), { createdAt: { $lte: createdBefore } });
        }
    }
    return filter;
}
exports.createStatisticsDatesFilter = createStatisticsDatesFilter;
function convertPropertyFilterToMongoFilter(propertyFilter) {
    const f = {
        uid: { $exists: true },
    };
    if (propertyFilter.uid) {
        f.uid = propertyFilter.uid;
    }
    if (propertyFilter.name) {
        f.name = propertyFilter.name;
    }
    if (propertyFilter.accountId) {
        f.accountIds = propertyFilter.accountId;
    }
    if (propertyFilter.isCompany !== undefined) {
        f.isCompany = propertyFilter.isCompany;
    }
    if (propertyFilter.parentUid) {
        f.parentUid = propertyFilter.parentUid;
    }
    return f;
}
//# sourceMappingURL=organizationModel.js.map