"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeProjectConfigs = void 0;
const object_1 = require("@signageos/lib/dist/Utils/object");
/**
 * It merges two project configs into a single one.
 * It prevents combining $project to contain both 0 and 1 values.
 * It doesn't make sense to have both, since ones are explicit includes and zeros are explicit excludes.
 *
 * E.g.:
 *
 * { name: 1, model: 1 } + { name: 0, firmwareVersion: 1 } = { model: 1, firmwareVersion: 1 }
 * The name field is removed, because it's both included and excluded. If the `name: 0` is kept, then MongoDB query would fail.
 *
 * However, when all project configs has only zeros, then the zeros are kept (it will use only excluding fields from the result).
 * { name: 0, model: 0 } + { firmwareVersion: 0 } = { name: 0, model: 0, firmwareVersion: 0 }
 */
function mergeProjectConfigs(projectConfig1, projectConfig2) {
    var _a;
    const result = {};
    const projectConfigs = [projectConfig1, projectConfig2];
    for (const projectConfig of projectConfigs) {
        if (projectConfig !== undefined) {
            for (const key of (0, object_1.getObjectKeys)(projectConfig)) {
                if (key in result) {
                    // The zero 0 is stronger than one 1.
                    // The existence of key in projectConfig is obvious. Non-null assertion is safe.
                    result[key] = Math.min(projectConfig[key], (_a = result[key]) !== null && _a !== void 0 ? _a : 0);
                }
                else {
                    result[key] = projectConfig[key];
                }
            }
        }
    }
    // Remove zeros when there are some ones.
    // It's because MongoDB doesn't support project explicit zero when there are explicit ones.
    // It doesn't make sense to have both, since ones are explicit includes and zeros are explicit excludes.
    if (Object.values(result).some((value) => value === 1)) {
        for (const key of (0, object_1.getObjectKeys)(result)) {
            // The _id field is an exception, because it's always included. It can be zero.
            if (key !== '_id' && result[key] === 0) {
                delete result[key];
            }
        }
    }
    if (Object.keys(result).length === 0) {
        return undefined;
    }
    return result;
}
exports.mergeProjectConfigs = mergeProjectConfigs;
//# sourceMappingURL=mergeProjectConfigs.js.map