"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createDeviceInfoModel = exports.prepareDeviceInfoTable = exports.deviceInfoCollection = exports.getDeviceFreeStorageStatusPercentage = void 0;
const _ = require("lodash");
const collections_1 = require("../../Lib/collections");
const versions_1 = require("../../../Helpers/versions");
const getDeviceFreeStorageStatusPercentage = (storageStatus) => {
    return Math.min(storageStatus.internal.capacity ? (storageStatus.internal.freeSpace / storageStatus.internal.capacity) * 100 : 100, storageStatus.removable.capacity ? (storageStatus.removable.freeSpace / storageStatus.removable.capacity) * 100 : 100);
};
exports.getDeviceFreeStorageStatusPercentage = getDeviceFreeStorageStatusPercentage;
const deviceInfoCollection = (conn) => conn.connection.collection(collections_1.DeviceCollection.Info, {
    writeConcern: {
        w: 1,
    },
});
exports.deviceInfoCollection = deviceInfoCollection;
const prepareDeviceInfoTable = (conn) => __awaiter(void 0, void 0, void 0, function* () {
    yield (0, exports.deviceInfoCollection)(conn).createIndex({ identityHash: 1 }, { name: 'identityHash', unique: true, background: true });
    yield (0, exports.deviceInfoCollection)(conn).createIndex({ minStorageStatusFreeSpacePercentage: 1 }, { name: 'minStorageStatusFreeSpacePercentage', background: true });
    yield (0, exports.deviceInfoCollection)(conn).createIndex({ 'networkInterfaces.ethernet.macAddress': 1 }, { name: 'networkInterfaces_ethernet_macAddress', background: true });
    yield (0, exports.deviceInfoCollection)(conn).createIndex({ 'networkInterfaces.ethernet.ipAddress': 1 }, { name: 'networkInterfaces_ethernet_ipAddress', background: true });
    yield (0, exports.deviceInfoCollection)(conn).createIndex({ 'networkInterfaces.wifi.macAddress': 1 }, { name: 'networkInterfaces_wifi_macAddress', background: true });
    yield (0, exports.deviceInfoCollection)(conn).createIndex({ 'networkInterfaces.wifi.ipAddress': 1 }, { name: 'networkInterfaces_wifi_ipAddress', background: true });
    yield (0, exports.deviceInfoCollection)(conn).createIndex({ managementPackageVersion: 1 }, { name: 'managementPackageVersion', background: true });
    yield (0, exports.deviceInfoCollection)(conn).createIndex({ frontPackageVersion: 1 }, { name: 'frontPackageVersion', background: true });
    yield (0, exports.deviceInfoCollection)(conn).createIndex({ managementPackageVersionNumber: 1 }, { name: 'managementPackageVersionNumber', background: true });
    yield (0, exports.deviceInfoCollection)(conn).createIndex({ frontPackageVersionNumber: 1 }, { name: 'frontPackageVersionNumber', background: true });
});
exports.prepareDeviceInfoTable = prepareDeviceInfoTable;
const createDeviceInfoModel = (conn) => ({
    create(identityHash) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash }, { $setOnInsert: { identityHash } }, { upsert: true });
        });
    },
    fetchByIdentityHash(identityHash) {
        return (0, exports.deviceInfoCollection)(conn).findOne({ identityHash });
    },
    fetchListByDevices(devices) {
        const query = (0, exports.deviceInfoCollection)(conn).find({
            identityHash: {
                $in: devices.map((device) => device.identityHash),
            },
        });
        return query.toArray();
    },
    fetchListByIdentityHashes(identityHashes, sorter, offset, limit) {
        const sortDirection = (sorter === null || sorter === void 0 ? void 0 : sorter.direction) === 'descending' ? -1 : 1;
        const pipeline = [{ $match: { identityHash: { $in: identityHashes } } }];
        switch (sorter === null || sorter === void 0 ? void 0 : sorter.field) {
            case 'storage':
                pipeline.push({ $sort: { 'storageStatus.internal.freeSpace': sortDirection } });
                break;
            case 'timezone':
                pipeline.push({ $sort: { 'currentTime.timezone': sortDirection } });
                break;
            case 'managementPackageVersion':
                pipeline.push({ $sort: { managementPackageVersionNumber: sortDirection } });
                break;
            case 'frontPackageVersion':
                pipeline.push({ $sort: { frontPackageVersionNumber: sortDirection } });
                break;
            default:
                break;
        }
        let aggregationCursor = (0, exports.deviceInfoCollection)(conn).aggregate(pipeline, { session: conn.session });
        if (offset) {
            aggregationCursor = aggregationCursor.skip(offset);
        }
        if (limit) {
            aggregationCursor = aggregationCursor.limit(limit);
        }
        return aggregationCursor.toArray();
    },
    fetchListByPropertyFilter(propertyFilter) {
        const filter = convertFilterToMongoFilter(propertyFilter);
        const pipeline = [{ $match: filter }];
        return (0, exports.deviceInfoCollection)(conn).aggregate(pipeline, { session: conn.session }).toArray();
    },
    fetchIdentityHashesByPropertyFilter(propertyFilter) {
        return __awaiter(this, void 0, void 0, function* () {
            const filter = convertFilterToMongoFilter(propertyFilter);
            const pipeline = [{ $match: filter }, { $project: { identityHash: 1 } }];
            const result = yield (0, exports.deviceInfoCollection)(conn).aggregate(pipeline, { session: conn.session }).toArray();
            return result.map((di) => di.identityHash);
        });
    },
    fetchIdentityHashesByStorageStatus(key, value) {
        const query = {
            ['minStorageStatusFreeSpacePercentage']: {
                [key === 'maxStorageStatusPercentage' ? '$gte' : '$lte']: 100 - value,
            },
        };
        return (0, exports.deviceInfoCollection)(conn)
            .find(query, { session: conn.session })
            .map((deviceInfo) => deviceInfo.identityHash)
            .toArray();
    },
    updateNetworkInterface(deviceInfoRow, networkSettings) {
        return __awaiter(this, void 0, void 0, function* () {
            const networkInterfaceSettings = {};
            const prefix = `networkInterfaces.${networkSettings.networkInterface}`;
            networkInterfaceSettings[`${prefix}.macAddress`] = networkSettings.macAddress;
            if (networkSettings.ipAddress !== undefined) {
                networkInterfaceSettings[`${prefix}.ipAddress`] = networkSettings.ipAddress;
            }
            if (networkSettings.netmask !== undefined) {
                networkInterfaceSettings[`${prefix}.netmask`] = networkSettings.netmask;
            }
            if (networkSettings.gateway !== undefined) {
                networkInterfaceSettings[`${prefix}.gateway`] = networkSettings.gateway;
            }
            if (networkSettings.domainNameServers !== undefined) {
                networkInterfaceSettings[`${prefix}.domainNameServers`] = networkSettings.domainNameServers;
            }
            if (networkSettings.interfaceName !== undefined) {
                networkInterfaceSettings[`${prefix}.interfaceName`] = networkSettings.interfaceName;
            }
            if (networkSettings.ssid !== undefined) {
                networkInterfaceSettings[`${prefix}.ssid`] = networkSettings.ssid;
            }
            if (networkSettings.signalStrength !== undefined) {
                networkInterfaceSettings[`${prefix}.signalStrength`] = networkSettings.signalStrength;
            }
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash: deviceInfoRow.identityHash }, {
                $set: networkInterfaceSettings,
                $setOnInsert: { identityHash: deviceInfoRow.identityHash },
            }, { upsert: true });
        });
    },
    updateSpecs(deviceInfoRow, specs) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash: deviceInfoRow.identityHash }, {
                $set: { specs },
                $setOnInsert: { identityHash: deviceInfoRow.identityHash },
            }, { upsert: true });
        });
    },
    updateBatteryStatus(deviceInfoRow, batteryStatus) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash: deviceInfoRow.identityHash }, {
                $set: { batteryStatus },
                $setOnInsert: { identityHash: deviceInfoRow.identityHash },
            }, { upsert: true });
        });
    },
    updateStorageStatus(deviceInfoRow, storageStatus) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash: deviceInfoRow.identityHash }, {
                $set: {
                    storageStatus,
                    minStorageStatusFreeSpacePercentage: (0, exports.getDeviceFreeStorageStatusPercentage)(storageStatus),
                },
                $setOnInsert: { identityHash: deviceInfoRow.identityHash },
            }, { upsert: true });
        });
    },
    updateCurrentTime(deviceInfoRow, currentTime, isIncorrect) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash: deviceInfoRow.identityHash }, {
                $set: Object.assign({ currentTime }, (isIncorrect !== undefined && { incorrectTime: isIncorrect })),
                $setOnInsert: { identityHash: deviceInfoRow.identityHash },
            }, { upsert: true });
        });
    },
    updateSupportedResolutions(deviceInfoRow, supportedResolutions) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash: deviceInfoRow.identityHash }, {
                $set: { supportedResolutions },
                $setOnInsert: { identityHash: deviceInfoRow.identityHash },
            }, { upsert: true });
        });
    },
    updatePolicyItemTypes(deviceInfoRow, policyItemTypes) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash: deviceInfoRow.identityHash }, {
                $set: { policyItemTypes },
                $setOnInsert: { identityHash: deviceInfoRow.identityHash },
            }, { upsert: true });
        });
    },
    updateProxy(deviceInfoRow, proxy) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash: deviceInfoRow.identityHash }, {
                $set: { proxy },
                $setOnInsert: { identityHash: deviceInfoRow.identityHash },
            }, { upsert: true });
        });
    },
    updateManagementPackageVersion(deviceIdentification, managementPackageVersion) {
        return __awaiter(this, void 0, void 0, function* () {
            const managementPackageVersionNumber = managementPackageVersion && (0, versions_1.isValid)(managementPackageVersion) ? (0, versions_1.numberizeStrictly)(managementPackageVersion) : null;
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash: deviceIdentification.identityHash }, {
                $set: {
                    managementPackageVersion,
                    managementPackageVersionNumber,
                },
            }, { session: conn.session });
        });
    },
    updateFrontPackageVersion(deviceIdentification, frontPackageVersion) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceInfoCollection)(conn).updateOne({ identityHash: deviceIdentification.identityHash }, {
                $set: {
                    frontPackageVersion,
                    frontPackageVersionNumber: frontPackageVersion && (0, versions_1.isValid)(frontPackageVersion) ? (0, versions_1.numberizeStrictly)(frontPackageVersion) : null,
                },
            }, { session: conn.session });
        });
    },
});
exports.createDeviceInfoModel = createDeviceInfoModel;
function convertFilterToMongoFilter(propertyFilter) {
    return Object.keys(propertyFilter)
        .filter((key) => !_.isNil(propertyFilter[key]))
        .reduce((accumulator, key) => {
        var _a, _b, _c, _d;
        switch (key) {
            case 'identityHashes':
                return Object.assign(Object.assign({}, accumulator), { identityHash: { $in: propertyFilter[key] } });
            case 'ethernetIpAddress':
                return Object.assign(Object.assign({}, accumulator), { $or: [...((_a = accumulator.$or) !== null && _a !== void 0 ? _a : []), { 'networkInterfaces.ethernet.ipAddress': propertyFilter[key] }] });
            case 'ethernetMacAddress':
                return Object.assign(Object.assign({}, accumulator), { $or: [...((_b = accumulator.$or) !== null && _b !== void 0 ? _b : []), { 'networkInterfaces.ethernet.macAddress': propertyFilter[key] }] });
            case 'wifiMacAddress':
                return Object.assign(Object.assign({}, accumulator), { $or: [...((_c = accumulator.$or) !== null && _c !== void 0 ? _c : []), { 'networkInterfaces.wifi.macAddress': propertyFilter[key] }] });
            case 'wifiIpAddress':
                return Object.assign(Object.assign({}, accumulator), { $or: [...((_d = accumulator.$or) !== null && _d !== void 0 ? _d : []), { 'networkInterfaces.wifi.ipAddress': propertyFilter[key] }] });
            case 'managementPackageVersion':
                return Object.assign(Object.assign({}, accumulator), { managementPackageVersion: { $eq: propertyFilter[key] } });
            case 'frontPackageVersion':
                return Object.assign(Object.assign({}, accumulator), { frontPackageVersion: { $eq: propertyFilter[key] } });
            default:
                throw new Error('Invalid property device info property filter.');
        }
    }, {});
}
//# sourceMappingURL=deviceInfoModel.js.map