"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createDeviceActionLogModel = exports.prepareDeviceActionLogTable = exports.deviceActionLogCollection = void 0;
const _ = require("lodash");
const escape_1 = require("@signageos/lib/dist/MongoDB/escape");
const generator_1 = require("@signageos/lib/dist/Hash/generator");
const deviceActionLogModel_1 = require("../../../Schema/Device/ActionLog/deviceActionLogModel");
const paginatorHelper_1 = require("../../../MongoDB/paginatorHelper");
const collections_1 = require("../../Lib/collections");
const STATUS_SUCCEEDED = 'succeeded';
const STATUS_FAILED = 'failed';
const STATUS_POSTPONED = 'postponed';
const deviceActionLogCollection = (conn) => conn.connection.collection(collections_1.DeviceCollection.ActionLog);
exports.deviceActionLogCollection = deviceActionLogCollection;
const prepareDeviceActionLogTable = (conn) => __awaiter(void 0, void 0, void 0, function* () {
    yield (0, exports.deviceActionLogCollection)(conn).createIndex({ id: 1 }, { name: 'id', unique: true, background: true });
    // By historical reason, it cannot be unique
    yield (0, exports.deviceActionLogCollection)(conn).createIndex({ uid: 1 }, { name: 'uid', background: true });
    yield (0, exports.deviceActionLogCollection)(conn).createIndex({ requestId: 1 }, { name: 'requestId', background: true });
    yield (0, exports.deviceActionLogCollection)(conn).createIndex({ uid: 1, type: 1 }, { name: 'uid_type', background: true });
    yield (0, exports.deviceActionLogCollection)(conn).createIndex({ deviceIdentityHash: 1, createdAt: 1 }, { name: 'deviceIdentiHash_createdAt', background: true });
    yield (0, exports.deviceActionLogCollection)(conn).createIndex({ deviceIdentityHash: 1, type: 1, createdAt: -1 }, { name: 'deviceIdentityHash_type_createdAt', background: true });
    yield (0, exports.deviceActionLogCollection)(conn).createIndex({ deviceIdentityHash: 1, type: 1, succeedAt: -1 }, { name: 'deviceIdentityHash_type_succeedAt', background: true });
});
exports.prepareDeviceActionLogTable = prepareDeviceActionLogTable;
const createDeviceActionLogModel = (conn) => ({
    fetchByUid(uid) {
        return (0, exports.deviceActionLogCollection)(conn).findOne({ uid }, { session: conn.session });
    },
    fetchByDevice(device) {
        return (0, exports.deviceActionLogCollection)(conn).findOne({ deviceIdentityHash: device.identityHash }, { session: conn.session });
    },
    fetchLatestResolvedByDevice(device, type) {
        const filter = {
            deviceIdentityHash: device.identityHash,
            type: deviceActionLogModel_1.DeviceActionType[type],
            $or: [{ succeedAt: { $exists: true } }, { failedAt: { $exists: true } }],
        };
        const options = {
            sort: { createdAt: -1 },
            session: conn.session,
        };
        return (0, exports.deviceActionLogCollection)(conn).findOne(filter, options);
    },
    /**
     * uid is not unique across types
     */
    fetchByUidAndType(uid, type) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield (0, exports.deviceActionLogCollection)(conn).findOne({ uid, type }, { session: conn.session });
        });
    },
    fetchListByIdentityHash(deviceIdentityHash, offset, limit) {
        let query = (0, exports.deviceActionLogCollection)(conn)
            .find({ deviceIdentityHash }, { session: conn.session })
            .sort({ createdAt: -1 });
        if (typeof offset !== 'undefined') {
            query = query.skip(offset);
        }
        if (typeof limit !== 'undefined') {
            query = query.limit(limit);
        }
        return query.toArray();
    },
    fetchListByDevice(device, type, filter) {
        let pipeline = [{ $match: { deviceIdentityHash: device.identityHash } }, { $match: { type: type } }];
        if (filter && (0, paginatorHelper_1.isPaginatorOfTimeType)(filter)) {
            if (filter === null || filter === void 0 ? void 0 : filter.since) {
                pipeline.push({ $match: { createdAt: { $gte: filter.since } } });
            }
            if (filter === null || filter === void 0 ? void 0 : filter.until) {
                pipeline.push({ $match: { createdAt: { $lt: filter.until } } });
            }
        }
        if (filter && (0, paginatorHelper_1.isPaginatorOfOffsetType)(filter)) {
            pipeline.push({ $skip: filter.offset });
        }
        if (filter) {
            if ((filter === null || filter === void 0 ? void 0 : filter.descending) === undefined || (filter === null || filter === void 0 ? void 0 : filter.descending) === true) {
                pipeline.push({ $sort: { [filter.sortKey]: -1 } });
            }
            else {
                pipeline.push({ $sort: { [filter.sortKey]: 1 } });
            }
        }
        if (filter === null || filter === void 0 ? void 0 : filter.limit) {
            pipeline.push({ $limit: filter.limit });
        }
        return (0, exports.deviceActionLogCollection)(conn).aggregate(pipeline, { session: conn.session }).toArray();
    },
    fetchAllListByDevice(device, filter) {
        let pipeline = [{ $match: { deviceIdentityHash: device.identityHash } }];
        if (filter && (0, paginatorHelper_1.isPaginatorOfTimeType)(filter)) {
            if (filter === null || filter === void 0 ? void 0 : filter.since) {
                pipeline.push({ $match: { createdAt: { $gte: filter.since } } });
            }
            if (filter === null || filter === void 0 ? void 0 : filter.until) {
                pipeline.push({ $match: { createdAt: { $lt: filter.until } } });
            }
        }
        if (filter && (0, paginatorHelper_1.isPaginatorOfOffsetType)(filter)) {
            pipeline.push({ $skip: filter.offset });
        }
        if (filter) {
            if ((filter === null || filter === void 0 ? void 0 : filter.descending) === undefined || (filter === null || filter === void 0 ? void 0 : filter.descending) === true) {
                pipeline.push({ $sort: { [filter.sortKey]: -1 } });
            }
            else {
                pipeline.push({ $sort: { [filter.sortKey]: 1 } });
            }
        }
        if (filter === null || filter === void 0 ? void 0 : filter.limit) {
            pipeline.push({ $limit: filter.limit });
        }
        return (0, exports.deviceActionLogCollection)(conn).aggregate(pipeline, { session: conn.session }).toArray();
    },
    create(uid, requestId, type, device, createdAt, data, originator) {
        return __awaiter(this, void 0, void 0, function* () {
            const id = (0, generator_1.generateUniqueHash)();
            // TODO remove type cast
            // I had to do it so that I don't change the code functionality
            // but it should be changed so that there's no type cast and tested
            const newRow = {
                id,
                uid,
                requestId,
                type: type,
                deviceIdentityHash: device.identityHash,
                createdAt: createdAt || new Date(),
                data: (0, escape_1.escapeObject)(data),
                originator: originator,
            };
            yield (0, exports.deviceActionLogCollection)(conn).replaceOne({ uid }, newRow, { session: conn.session, upsert: true });
        });
    },
    markSucceed(deviceActionLogRow, succeedAt) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceActionLogCollection)(conn).updateOne({ requestId: deviceActionLogRow.requestId }, { $set: { succeedAt } }, { session: conn.session });
        });
    },
    markFailed(deviceActionLogRow, failedAt) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceActionLogCollection)(conn).updateOne({ requestId: deviceActionLogRow.requestId }, { $set: { failedAt } }, { session: conn.session });
        });
    },
    updateData(deviceActionLogRow, updatedData, updatedAt) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.deviceActionLogCollection)(conn).updateOne({ requestId: deviceActionLogRow.requestId }, {
                $set: {
                    updatedAt,
                    data: (0, escape_1.escapeObject)(updatedData),
                },
            }, { session: conn.session });
        });
    },
    fetchLatestSucceededByDevice(device, type) {
        const filter = {
            deviceIdentityHash: device.identityHash,
            type: deviceActionLogModel_1.DeviceActionType[type],
            succeedAt: { $exists: true },
        };
        const options = {
            sort: { succeedAt: -1 },
            session: conn.session,
        };
        return (0, exports.deviceActionLogCollection)(conn).findOne(filter, options);
    },
    fetchFailedByDeviceList(deviceList, type, from) {
        const failedFilter = from ? { failedAt: { $gte: from } } : { failedAt: { $exists: true } };
        const filter = Object.assign({ type: deviceActionLogModel_1.DeviceActionType[type], deviceIdentityHash: {
                $in: deviceList.map((d) => d.identityHash),
            } }, failedFilter);
        return (0, exports.deviceActionLogCollection)(conn).find(filter, { session: conn.session }).toArray();
    },
    fetchListByDevices(devices, type) {
        const filter = {
            type,
            deviceIdentityHash: {
                $in: devices.map((d) => d.identityHash),
            },
        };
        return (0, exports.deviceActionLogCollection)(conn).find(filter, { session: conn.session }).toArray();
    },
    fetchLatestListByDevices(devices, type, limit) {
        const filter = {
            type,
            deviceIdentityHash: { $in: devices.map((d) => d.identityHash) },
        };
        return (0, exports.deviceActionLogCollection)(conn)
            .find(filter, { session: conn.session })
            .sort({ createdAt: -1 })
            .limit(limit ? limit : 0)
            .toArray();
    },
    fetchLatestGroupedSucceededList(device, type, dataPropertyName) {
        return (0, exports.deviceActionLogCollection)(conn)
            .aggregate([
            { $match: { deviceIdentityHash: device.identityHash, type, succeedAt: { $ne: null } } },
            { $sort: { succeedAt: -1 } },
            { $group: { _id: `$data.${dataPropertyName}`, log: { $first: '$$ROOT' } } },
            { $replaceRoot: { newRoot: '$log' } },
        ])
            .toArray();
    },
    fetchListByPropertyFilter(propertyFilter, type) {
        const filter = convertPropertyFilterToMongoFilter(propertyFilter, { type });
        return (0, exports.deviceActionLogCollection)(conn)
            .find(filter, { session: conn.session, sort: { createdAt: -1 } })
            .sort({ createdAt: -1 })
            .toArray();
    },
    fetchListByIdentityHashAndPropertyFilter(deviceIdentityHash, propertyFilter, offset, limit) {
        var _a, _b, _c;
        return __awaiter(this, void 0, void 0, function* () {
            const filter = convertPropertyFilterToMongoFilter(propertyFilter, { deviceIdentityHash });
            const result = yield (0, exports.deviceActionLogCollection)(conn)
                .aggregate([
                {
                    $match: { deviceIdentityHash: deviceIdentityHash },
                },
                {
                    $facet: {
                        actionLog: [
                            {
                                $match: filter,
                            },
                            { $sort: { createdAt: -1 } },
                            ...(typeof offset !== 'undefined' ? [{ $skip: offset }] : []),
                            ...(typeof limit !== 'undefined' ? [{ $limit: limit }] : []),
                        ],
                        totalCount: [
                            {
                                $match: filter,
                            },
                            {
                                $count: 'count',
                            },
                        ],
                    },
                },
            ])
                .toArray();
            return {
                actionLog: result[0].actionLog,
                totalCount: (_c = (_b = (_a = result[0].totalCount) === null || _a === void 0 ? void 0 : _a[0]) === null || _b === void 0 ? void 0 : _b.count) !== null && _c !== void 0 ? _c : 0,
            };
        });
    },
    fetchLatestByDevice(device, type) {
        const latestFilter = {
            deviceIdentityHash: device.identityHash,
            type: deviceActionLogModel_1.DeviceActionType[type],
        };
        const options = {
            session: conn.session,
            sort: { createdAt: -1 },
        };
        return (0, exports.deviceActionLogCollection)(conn).findOne(latestFilter, options);
    },
    countFailedByDevice(device, type, from) {
        const failedFilter = from ? { failedAt: { $gte: from } } : { failedAt: { $exists: true } };
        const filter = Object.assign({ deviceIdentityHash: device.identityHash, type: deviceActionLogModel_1.DeviceActionType[type] }, failedFilter);
        return (0, exports.deviceActionLogCollection)(conn).find(filter, { session: conn.session }).count();
    },
    fetchListByDeviceAndStatus(// specific stuff: from deviceDateTimeModel
    device, type, status, limit) {
        if (!isStatusValid(status)) {
            throw new Error(`Unknown status value: "${type}"`);
        }
        let filter = {
            deviceIdentityHash: device.identityHash,
            type: deviceActionLogModel_1.DeviceActionType[type],
        };
        switch (status) {
            case STATUS_SUCCEEDED:
                filter = Object.assign(Object.assign({}, filter), { succeedAt: { $exists: true } });
                break;
            case STATUS_FAILED:
                filter = Object.assign(Object.assign({}, filter), { failedAt: { $exists: true } });
                break;
            case STATUS_POSTPONED:
                return Promise.reject(Error(`Do not use that 'postponed', it's deprecated`));
            case undefined:
            default:
                break;
        }
        return (0, exports.deviceActionLogCollection)(conn)
            .find(filter, { session: conn.session })
            .sort({ createdAt: -1 })
            .limit(limit ? limit : 0)
            .toArray();
    },
});
exports.createDeviceActionLogModel = createDeviceActionLogModel;
function convertPropertyFilterToMongoFilter(propertyFilter, carry) {
    return Object.keys(propertyFilter)
        .filter((key) => !_.isUndefined(propertyFilter[key]))
        .reduce((accumulator, key) => {
        switch (key) {
            case 'deviceIdentityHash':
                return Object.assign(Object.assign({}, accumulator), { deviceIdentityHash: propertyFilter[key] });
            case 'requestId':
                return Object.assign(Object.assign({}, accumulator), { requestId: propertyFilter[key] });
            case 'failedAt':
                return Object.assign(Object.assign({}, accumulator), { failedAt: propertyFilter[key] });
            case 'succeedAt':
                return Object.assign(Object.assign({}, accumulator), { succeedAt: propertyFilter[key] });
            case 'createdAt':
                return Object.assign(Object.assign({}, accumulator), { createdAt: propertyFilter[key] });
            case 'type':
                return Object.assign(Object.assign({}, accumulator), { type: propertyFilter[key] });
            case 'uid':
                return Object.assign(Object.assign({}, accumulator), { uid: propertyFilter[key] });
            case 'since':
                return Object.assign(Object.assign({}, accumulator), { createdAt: Object.assign(Object.assign({}, accumulator.createdAt), { $gte: propertyFilter[key] }) });
            case 'until':
                return Object.assign(Object.assign({}, accumulator), { createdAt: Object.assign(Object.assign({}, accumulator.createdAt), { $lte: propertyFilter[key] }) });
            case 'originator':
                return Object.assign(Object.assign({}, accumulator), { originator: propertyFilter[key] });
            case 'triggeredBy':
                return Object.assign(Object.assign(Object.assign(Object.assign({}, accumulator), (propertyFilter[key] === 'user'
                    ? { 'originator.accountId': { $exists: true }, 'originator.organizationUid': { $exists: false } }
                    : {})), (propertyFilter[key] === 'api'
                    ? { 'originator.accountId': { $exists: false }, 'originator.organizationUid': { $exists: true } }
                    : {})), (propertyFilter[key] === 'system'
                    ? {
                        'originator.accountId': {
                            $exists: false,
                        },
                        'originator.organizationUid': {
                            $exists: false,
                        },
                    }
                    : {}));
            case 'data':
            default:
                throw new Error('Invalid prop device action log prop filter.');
        }
    }, carry);
}
function isStatusValid(status) {
    if (!status) {
        return true;
    }
    if (status === STATUS_SUCCEEDED || status === STATUS_FAILED || status === STATUS_POSTPONED) {
        return true;
    }
    return false;
}
//# sourceMappingURL=deviceActionLogModel.js.map