"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createDeviceMonitoringLogModel = void 0;
const influxdb_client_1 = require("@influxdata/influxdb-client");
const helper_1 = require("./helper");
const expressionCreator_1 = require("../../../Helpers/expressionCreator");
const createDeviceMonitoringLogModel = (conn) => {
    function createPointData(point, type, id, data) {
        return helper_1.monitoringPointCreator.create(point, type, id, data);
    }
    return {
        create(id, type, device, createdAt, data) {
            return __awaiter(this, void 0, void 0, function* () {
                const monitoringLog = {
                    id,
                    deviceIdentityHash: device.identityHash,
                    organizationUid: device.organizationUid === null ? undefined : device.organizationUid,
                    type,
                    createdAt,
                    data,
                };
                yield this.createMany([monitoringLog]);
            });
        },
        createMany(monitoringLogs) {
            return __awaiter(this, void 0, void 0, function* () {
                const writeApi = conn.getClient().getWriteApi(conn.getOrganization(), conn.getBucket());
                const points = monitoringLogs.map((mLog) => {
                    const point = new influxdb_client_1.Point(mLog.type).tag('deviceIdentityHash', mLog.deviceIdentityHash).timestamp(mLog.createdAt);
                    const pointWithOptionalOrganization = mLog.organizationUid ? point.tag('organizationUid', mLog.organizationUid) : point;
                    const dataPoint = createPointData(pointWithOptionalOrganization, mLog.type, mLog.id, mLog.data);
                    return dataPoint;
                });
                writeApi.writePoints(points);
                yield writeApi.close();
            });
        },
        fetchById(_id) {
            return __awaiter(this, void 0, void 0, function* () {
                throw new Error(`Not implemented method.`);
            });
        },
        fetchLatestByDeviceAndType(device, type) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const queryable = (0, helper_1.createQueryable)(type);
                const query = queryable.getLastValueQuery(conn.getBucket(), device.identityHash);
                const rows = yield queryApi.collectRows(query);
                const pivotedRows = queryable.transformPivoted(rows);
                const lastPivotedRow = pivotedRows[0];
                return lastPivotedRow !== null && lastPivotedRow !== void 0 ? lastPivotedRow : null;
            });
        },
        fetchLatestByIdentityHashAndType(_identityHash, _type) {
            return __awaiter(this, void 0, void 0, function* () {
                throw new Error('Not implemented');
            });
        },
        fetchLastNValuesByDeviceAndType(device, type, n) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const queryable = (0, helper_1.createQueryable)(type);
                const query = queryable.getLastNValuesQuery(conn.getBucket(), device.identityHash, n);
                const rows = yield queryApi.collectRows(query);
                return queryable.transformPivoted(rows);
            });
        },
        fetchListByDevice(device, type, filter = {}) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const queryable = (0, helper_1.createQueryable)(type);
                const query = queryable.getValuesQuery(conn.getBucket(), device.identityHash, filter);
                const rows = yield queryApi.collectRows(query);
                return queryable.transformPivoted(rows);
            });
        },
        /**
         * Don't use to fetch data from measurements which could contain tens of thousands or more points for given time window.
         */
        fetchListByDeviceAndTypeSince(device, type, since) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const queryable = (0, helper_1.createQueryable)(type);
                const query = queryable.getValuesSinceQuery(conn.getBucket(), device.identityHash, since);
                const rows = yield queryApi.collectRows(query);
                return queryable.transformPivoted(rows);
            });
        },
        invalidate() {
            return __awaiter(this, void 0, void 0, function* () {
                throw new Error(`Not implemented method.`);
            });
        },
        fetchUptimeByOrganization(organizationUid, start, stop) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const fluxStart = (0, influxdb_client_1.fluxDateTime)(start.toISOString());
                const fluxStop = (0, influxdb_client_1.fluxDateTime)(stop.toISOString());
                const fluxQuery = calculateDeviceUptime(conn.getBucket(), 'organizationUid', organizationUid, fluxStart, fluxStop);
                const rows = (yield queryApi.collectRows(fluxQuery.toString()));
                return rows[0];
            });
        },
        fetchUptimeByIdentityHash(identityHash, start, stop) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const fluxStart = (0, influxdb_client_1.fluxDateTime)(start.toISOString());
                const fluxStop = (0, influxdb_client_1.fluxDateTime)(stop.toISOString());
                const fluxQuery = calculateDeviceUptime(conn.getBucket(), 'deviceIdentityHash', identityHash, fluxStart, fluxStop);
                const rows = (yield queryApi.collectRows(fluxQuery.toString()));
                return rows[0];
            });
        },
        fetchDevicesStatusCount(start, stop, organizationUids) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const multipleOrExpression = (0, expressionCreator_1.createEqualsMultipleOrExpression)('r', 'organizationUid', organizationUids);
                const fluxQuery = (0, influxdb_client_1.flux) `
				from(bucket: ${conn.getBucket()})
					|> range(start: ${start}, stop: ${stop})
					|> filter(fn: (r) => r["_measurement"] == "ONLINE_STATUS"
						and r._field == "online"
						and (${(0, influxdb_client_1.fluxExpression)(multipleOrExpression)})
					)
					|> last()
					|> group()
					|> reduce(
						fn: (r, accumulator) => ({
							onlineCount: if r._value == true then accumulator.onlineCount + 1 else accumulator.onlineCount,
							offlineCount: if r._value == false then accumulator.offlineCount + 1 else accumulator.offlineCount,
							count: accumulator.count + 1
						}),
						identity: { onlineCount: 0, offlineCount: 0, count: 0 }
					)
			`;
                const rows = (yield queryApi.collectRows(fluxQuery.toString()));
                return rows[0];
            });
        },
        fetchOfflineDeviceIdentityHashes(start, stop, organizationUids, offlineForMoreThan) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const multipleOrExpression = (0, expressionCreator_1.createEqualsMultipleOrExpression)('r', 'organizationUid', organizationUids);
                const fluxQuery = (0, influxdb_client_1.flux) `
				from(bucket: ${conn.getBucket()})
					|> range(start: ${start}, stop: ${stop})
					|> filter(fn: (r) => r["_measurement"] == "ONLINE_STATUS"
						and r._field == "online"
						and (${(0, influxdb_client_1.fluxExpression)(multipleOrExpression)})
					)
					|> group(columns: ["deviceIdentityHash"])
					|> last()
					|> group()
					|> filter(fn: (r) => r["_value"] == false and ((int(v: now()) - int(v: r["_time"])) / int(v: 10^9)) > ${offlineForMoreThan !== null && offlineForMoreThan !== void 0 ? offlineForMoreThan : 0})
  					|> map(fn: (r) => ({ deviceIdentityHash: r.deviceIdentityHash }))
			`;
                const rows = (yield queryApi.collectRows(fluxQuery.toString()));
                return rows;
            });
        },
        fetchLatelyWentOnlineDeviceIdentityHashes(start, stop, organizationUids, beforeOfflineAtLeast) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const multipleOrExpression = (0, expressionCreator_1.createEqualsMultipleOrExpression)('r', 'organizationUid', organizationUids);
                const startDate = new Date(0);
                const fluxQuery = (0, influxdb_client_1.flux) `
				from(bucket: ${conn.getBucket()})
					|> range(start: ${startDate}, stop: ${stop})
					|> filter(fn: (r) => r["_measurement"] == "ONLINE_STATUS"
						and r._field == "online"
						and (${(0, influxdb_client_1.fluxExpression)(multipleOrExpression)})
					)
					|> group(columns: ["deviceIdentityHash"])
					|> elapsed()
					|> filter(fn: (r) => r["_time"] > ${start})
					|> reduce(
						fn: (r, accumulator) => {
							tmpTime = uint(v: r._time)
							tmpElapsed = uint(v: r.elapsed)
							return {
								lastOnline: if r._value == true and tmpTime > accumulator.lastOnline then tmpTime else accumulator.lastOnline,
								offlineTime:  if r._value == true and tmpTime > accumulator.lastOnline then tmpElapsed else accumulator.offlineTime,
								lastOffline: if r._value == false and tmpTime > accumulator.lastOffline then tmpTime else accumulator.lastOffline
							}
						},
						identity: { lastOnline: uint(v: 0), lastOffline: uint(v: 0), offlineTime: uint(v: 0) }
					)
					|> filter(fn: (r) => r.lastOnline > r.lastOffline and r.offlineTime > ${beforeOfflineAtLeast} )
					|> map(fn: (r) => ({ deviceIdentityHash: r.deviceIdentityHash }))
			`;
                const rows = (yield queryApi.collectRows(fluxQuery.toString()));
                return rows;
            });
        },
        fetchLatestByDevicesAndType(deviceIdentityHashes, type) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const queryable = (0, helper_1.createQueryable)(type);
                const query = queryable.getLastValuesSinceQuery(conn.getBucket(), deviceIdentityHashes, new Date(0));
                const rows = yield queryApi.collectRows(query);
                return queryable.transformPivoted(rows);
            });
        },
        fetchLatestListByDevices() {
            return __awaiter(this, void 0, void 0, function* () {
                throw new Error('Not implemented');
            });
        },
        fetchMonitoringLogListByDevice(device, type, since, until, limit) {
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const queryable = (0, helper_1.createQueryable)(type);
                const query = queryable.getValuesQueryPivoted(conn.getBucket(), device.identityHash, since, until, limit);
                const rows = yield queryApi.collectRows(query);
                return queryable.transformPivoted(rows);
            });
        },
        fetchList(params) {
            var _a, _b, _c, _d, _e, _f;
            return __awaiter(this, void 0, void 0, function* () {
                const queryApi = conn.getClient().getQueryApi(conn.getOrganization());
                const start = (_b = (_a = params === null || params === void 0 ? void 0 : params.filter) === null || _a === void 0 ? void 0 : _a.since) !== null && _b !== void 0 ? _b : new Date(0);
                const stop = (_d = (_c = params === null || params === void 0 ? void 0 : params.filter) === null || _c === void 0 ? void 0 : _c.until) !== null && _d !== void 0 ? _d : new Date();
                const queryable = (0, helper_1.createQueryable)(params.filter.type);
                let sortExpression = '';
                if (((_e = params.sort) === null || _e === void 0 ? void 0 : _e.field) === 'createdAt') {
                    const desc = params.sort.order === 'descending';
                    sortExpression = `|> sort(columns: ["_time"], desc: ${desc ? (0, influxdb_client_1.fluxBool)(true) : (0, influxdb_client_1.fluxBool)(false)})`;
                }
                let limitExpression = '';
                if (params.pagination) {
                    limitExpression = `|> limit(n: ${(0, influxdb_client_1.fluxInteger)((_f = params.pagination) === null || _f === void 0 ? void 0 : _f.limit)})`;
                }
                const fluxQuery = (0, influxdb_client_1.flux) `
				from(bucket: ${conn.getBucket()})
					|> range(start: ${start}, stop: ${stop})
					|> filter(fn: (r) => r["deviceIdentityHash"] == ${(0, influxdb_client_1.fluxString)(params.filter.deviceIdentityHash)}
						and r["_measurement"] == ${(0, influxdb_client_1.fluxString)(params.filter.type)}
					)
					|> pivot(rowKey:["_time"], columnKey: ["_field"], valueColumn: "_value")
					|> group()
					${(0, influxdb_client_1.fluxExpression)(sortExpression)}
					|> drop(columns: ["_stop", "_start"])
					|> rename(columns: { _time: "createdAt", _measurement: "type" })
					${(0, influxdb_client_1.fluxExpression)(limitExpression)}
			`;
                const query = fluxQuery.toString();
                const rows = yield queryApi.collectRows(query);
                return queryable.transformPivoted(rows);
            });
        },
    };
};
exports.createDeviceMonitoringLogModel = createDeviceMonitoringLogModel;
function calculateDeviceUptime(bucket, identifierName, identifierValue, fluxStart, fluxStop) {
    return (0, influxdb_client_1.flux) `
		firstPointBeforeInterval = from(bucket: ${bucket})
			|> range(start: 0, stop: ${fluxStart})
			|> filter(fn: (r) => r["_measurement"] == "ONLINE_STATUS"
				and r._field == "online"
				and r[${(0, influxdb_client_1.fluxString)(identifierName)}] == ${(0, influxdb_client_1.fluxString)(identifierValue)}
			)
			|> last()
			|> map(fn: (r) => ({ r with _time: ${fluxStart} }))

		interval = from(bucket: ${bucket})
			|> range(start: ${fluxStart}, stop: ${fluxStop})
			|> filter(fn: (r) => r["_measurement"] == "ONLINE_STATUS"
				and r._field == "online"
				and r[${(0, influxdb_client_1.fluxString)(identifierName)}] == ${(0, influxdb_client_1.fluxString)(identifierValue)}
			)

		union(tables: [firstPointBeforeInterval, interval])
			|> group(columns: ["deviceIdentityHash"])
			|> sort(columns: ["_time"])

			|> reduce(fn: (r, accumulator) => {
					prevOnline = if r._value == true and (accumulator.prevState == 0 or accumulator.prevState == -1)
						then uint(v: r._time)
						else accumulator.prevOnline

					prevOffline = if r._value == false and (accumulator.prevState == 1 or accumulator.prevState == -1)
						then uint(v: r._time)
						else accumulator.prevOffline

					toAdd = if r._value == false and accumulator.prevState == 1
						then uint(v: r._time) - accumulator.prevOnline
						else uint(v: 0)

					toAddOffline = if r._value == true and accumulator.prevState == 0
						then uint(v: r._time) - accumulator.prevOffline
						else uint(v: 0)
					return {
						uptime: toAdd + accumulator.uptime,
						prevState: if r._value == false then 0 else 1,
						prevOnline: prevOnline,
						prevOffline: prevOffline,
						downtime: toAddOffline + accumulator.downtime
					}
				},
					identity: { uptime: uint(v: 0), downtime: uint(v: 0), prevState: -1, prevOnline: uint(v: 0), prevOffline: uint(v: 0) }
				)
			|> reduce(fn: (r, accumulator) => {
					uptime = if r.prevState == 1 then r.uptime + ( uint(v: ${fluxStop}) - r.prevOnline ) else r.uptime
					downtime = if r.prevState == 0 then r.downtime + ( uint(v: ${fluxStop}) - r.prevOffline ) else r.downtime
					return {
						uptime: uptime,
						downtime: downtime
					}
				},
				identity: { uptime: uint(v: 0), downtime: uint(v: 0) }
			)
			|> group()
			|> reduce(
				fn: (r, accumulator) => ({
					uptime: accumulator.uptime + (r.uptime / uint(v: 1000000000)),
					downtime: accumulator.downtime + (r.downtime / uint(v: 1000000000))
				}),
				identity: { uptime: uint(v: 0), downtime: uint(v: 0) }
			)
	`;
}
//# sourceMappingURL=deviceMonitoringLogModel.js.map