"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.semverComparatorDesc = exports.numberizeStrictlyBigInt = exports.numberizeStrictly = exports.numberizeBigInt = exports.numberize = exports.validate = exports.isValid = void 0;
const enums_1 = require("./enums");
function isValid(version) {
    const parts = version.split(/[-+]/);
    const base = parts[0].split('.');
    if (base.length !== 3) {
        return false;
    }
    for (let part of base) {
        if (part.length === 0 || part.length > 3) {
            return false;
        }
    }
    return true;
}
exports.isValid = isValid;
function validate(version) {
    if (!isValid(version)) {
        throw new Error(`Invalid semantic version given: "${version}"`);
    }
}
exports.validate = validate;
function numberize(version) {
    const parts = version.split(/[-+]/);
    const base = parts[0];
    let padded = [];
    for (let v of base.split('.')) {
        padded.push(v.padStart(3, '0'));
    }
    // to ensure that 2.1.3 is sorted above 2.1.3-rc.4 put large decimal offset to version without rc flag
    const decimalPart = parts.length > 1 ? stringToNumber(version.substring(base.length)) : enums_1.semverDecimalDefault;
    return parseFloat(`${padded.join('')}.${decimalPart}`);
}
exports.numberize = numberize;
function numberizeBigInt(version) {
    return Math.round(numberize(version) * 1e6);
}
exports.numberizeBigInt = numberizeBigInt;
function numberizeStrictly(version) {
    validate(version);
    return numberize(version);
}
exports.numberizeStrictly = numberizeStrictly;
function numberizeStrictlyBigInt(version) {
    validate(version);
    return numberizeBigInt(version);
}
exports.numberizeStrictlyBigInt = numberizeStrictlyBigInt;
// returns the sum of ascii codes used in the string
function stringToNumber(str) {
    let sum = 0;
    for (let i = 0; i < str.length; i++) {
        sum += str.charCodeAt(i);
    }
    return sum;
}
/**
 * @T type with semver `version` property
 * @throws `Error` instance if `version` is not semver-like
 * @see `numberizeStrictly`
 * @param a semver-like object
 * @param b semver-like object
 * @returns -1, 0, 1
 */
function semverComparatorDesc(a, b) {
    const numberizedA = numberizeStrictly(a.version);
    const numberizedB = numberizeStrictly(b.version);
    return numberizedA < numberizedB ? 1 : numberizedA > numberizedB ? -1 : 0;
}
exports.semverComparatorDesc = semverComparatorDesc;
//# sourceMappingURL=versions.js.map