#!/usr/bin/env node

const semver = require('semver');
const fs = require('fs');

const packagePath = process.cwd() + '/package.json';
const package = require(packagePath);

const ignorePath = process.cwd() + '/.check-depsignore';
let ignore = [];
try {
	ignore = fs.readFileSync(ignorePath).toString().split('\n');
	console.log(`Ignoring packages from ${ignorePath}`);
} catch (_error) {
	console.log('No ignore file found');
}

const runControlPath = process.cwd() + '/.check-depsrc.json';
let runControl = {};
try {
	runControl = require(runControlPath);
	console.log(`Using run control from ${runControlPath}`);
	if (runControl.ignore) {
		ignore = ignore.concat(runControl.ignore);
	}
} catch (error) {
	console.log('No Run Control file found');
}

const includeRegExp = new RegExp(process.argv[2] || '.+');
const excludeRegExp = new RegExp(process.argv[3] || '^$');

const deps = { ...(package.dependencies || {}), ...(package.devDependencies || {}) };

const depNames = Object.keys(deps);

const matchedDepNames = depNames.filter((depName) => includeRegExp.test(depName) && !excludeRegExp.test(depName));

const prereleaseDepNames = [];
const inexactDepNames = [];
const expectedRangeDepNames = [];

function isExpectedRangeValid(depName, version) {
	return version.startsWith(runControl.expectRanges[depName]);
}

for (const depName of matchedDepNames) {
	if (ignore.includes(depName)) {
		continue;
	}
	const depVersion = deps[depName];
	if (semver.prerelease(semver.minVersion(depVersion)) !== null) {
		prereleaseDepNames.push(depName);
	}
	if (runControl.expectRanges && depName in runControl.expectRanges) {
		if (!isExpectedRangeValid(depName, depVersion)) {
			expectedRangeDepNames.push(depName);
		}
	} else {
		if (!semver.valid(depVersion)) {
			// Check inexact only when not expected range
			inexactDepNames.push(depName);
		}
	}
}

if (prereleaseDepNames.length > 0) {
	const prereleaseDeps = {};
	prereleaseDepNames.forEach((depName) => {
		prereleaseDeps[depName] = deps[depName];
	});
	console.error(`Some packages has prerelease tags in version in package.json deps`, prereleaseDepNames.join(', '), prereleaseDeps);
}

if (expectedRangeDepNames.length > 0) {
	const expectedRangeDeps = {};
	expectedRangeDepNames.forEach((depName) => {
		expectedRangeDeps[depName] = deps[depName];
	});
	console.error(
		`Some packages has expectation of range of versions specified in package.json deps`,
		expectedRangeDepNames.map((depName) => `${depName}->"${runControl.expectRanges[depName]}"`).join(', '),
		expectedRangeDeps,
	);
}

if (inexactDepNames.length > 0) {
	const inexactDeps = {};
	inexactDepNames.forEach((depName) => {
		inexactDeps[depName] = deps[depName];
	});
	console.error(`Some packages has not exact version specified in package.json deps`, inexactDepNames.join(', '), inexactDeps);
}

if (prereleaseDepNames.length > 0 || expectedRangeDepNames.length > 0 || inexactDepNames.length > 0) {
	process.exit(1);
}
