"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAutoWsSocketServer = void 0;
const wsServerFactory_1 = require("./wsServerFactory");
const wait_1 = require("../Timer/wait");
const Debug = require("debug");
const debug = Debug('@signageos/lib:WebSocket:autoWsServerFactory');
function createAutoWsSocketServer(httpServer, options = {}) {
    const wsSocketWrapper = (0, wsServerFactory_1.createWsSocketServer)(httpServer, options);
    let resolveDisconnectedPromises = [];
    let messagesQueue = [];
    let messagesExpectingResponseQueue = [];
    let boundMessageListeners = [];
    let forceDisconnected = false;
    let socketWasConnected = false;
    let currentSocket = null;
    const wrappedSocket = {
        getDisconnectedPromise() {
            return __awaiter(this, void 0, void 0, function* () {
                return new Promise((resolve) => resolveDisconnectedPromises.push(resolve));
            });
        },
        bindError(_listener) {
            // never error
            return () => undefined;
        },
        bindMessage(event, listener) {
            const boundMessageListener = { event, listener };
            boundMessageListeners.push(boundMessageListener);
            debug('Number of listeners', boundMessageListeners.length);
            let removeListener;
            if (currentSocket) {
                removeListener = currentSocket.bindMessage(event, listener);
            }
            return () => {
                if (removeListener) {
                    removeListener();
                }
                boundMessageListeners.splice(boundMessageListeners.indexOf(boundMessageListener), 1);
            };
        },
        getMessagePromise(event) {
            return new Promise((resolve) => {
                const unbindMessage = this.bindMessage(event, (message) => {
                    unbindMessage();
                    resolve(message);
                });
            });
        },
        sendMessage(event, payload) {
            return __awaiter(this, void 0, void 0, function* () {
                if (currentSocket) {
                    try {
                        yield Promise.race([currentSocket.sendMessage(event, payload), (0, wait_1.default)(5e3).then(() => Promise.reject(`Send message timed-out`))]);
                    }
                    catch (error) {
                        debug('Failed to send message', event, payload, error);
                        messagesQueue.push({ event, payload });
                        debug('Number of messages in queue', messagesQueue.length);
                        try {
                            yield currentSocket.disconnect();
                        }
                        catch (error2) {
                            debug('nothing when already disconnected', event, payload, error2);
                        }
                    }
                }
                else {
                    debug('Requeued message', event, payload);
                    messagesQueue.push({ event, payload });
                    debug('Number of messages in queue', messagesQueue.length);
                }
            });
        },
        sendMessageExpectingResponse(event, payload) {
            return __awaiter(this, void 0, void 0, function* () {
                if (currentSocket) {
                    try {
                        yield Promise.race([
                            currentSocket.sendMessageExpectingResponse(event, payload),
                            (0, wait_1.default)(5e3).then(() => Promise.reject(`Send message expecting response timed-out`)),
                        ]);
                    }
                    catch (error) {
                        debug('Failed to send message expecting response', event, payload, error);
                        messagesExpectingResponseQueue.push({ event, payload });
                        debug('Number of messages in queue expecting response', messagesExpectingResponseQueue.length);
                        try {
                            yield currentSocket.disconnect();
                        }
                        catch (error2) {
                            debug('nothing when already disconnected', event, payload, error2);
                        }
                    }
                }
                else {
                    debug('Requeued message expecting response', event, payload);
                    messagesExpectingResponseQueue.push({ event, payload });
                    debug('Number of messages in queue expecting response', messagesExpectingResponseQueue.length);
                }
            });
        },
        disconnect(code = 1000, reason) {
            return __awaiter(this, void 0, void 0, function* () {
                if (forceDisconnected) {
                    throw new Error(`The socket is already disconnected`);
                }
                forceDisconnected = true;
                if (currentSocket) {
                    yield currentSocket.disconnect(code, reason);
                }
                else {
                    resolveDisconnectedPromises.forEach((resolve) => resolve({ code, reason }));
                }
            });
        },
        drain() {
            if (!forceDisconnected) {
                throw new Error(`Disconnect socket first`);
            }
        },
        get hostname() {
            return currentSocket === null || currentSocket === void 0 ? void 0 : currentSocket.hostname;
        },
        get path() {
            return currentSocket === null || currentSocket === void 0 ? void 0 : currentSocket.path;
        },
        get headers() {
            return currentSocket === null || currentSocket === void 0 ? void 0 : currentSocket.headers;
        },
    };
    function reinitiateSocket() {
        return __awaiter(this, void 0, void 0, function* () {
            if (currentSocket) {
                for (const boundMessageListener of boundMessageListeners) {
                    currentSocket.bindMessage(boundMessageListener.event, boundMessageListener.listener);
                }
            }
            // Wait a bit before queued messages are resend to allow client bind message events
            yield (0, wait_1.default)(20);
            let message;
            while ((message = messagesQueue.shift()) && currentSocket !== null) {
                yield wrappedSocket.sendMessage(message.event, message.payload);
            }
            debug('Number of messages in queue after repeat', messagesQueue.length);
            while ((message = messagesExpectingResponseQueue.shift()) && currentSocket !== null) {
                yield wrappedSocket.sendMessageExpectingResponse(message.event, message.payload);
            }
            debug('Number of messages in queue expecting response after repeat', messagesExpectingResponseQueue.length);
        });
    }
    return {
        server: {
            bindConnection(serverListener) {
                wsSocketWrapper.server.bindConnection((socket) => {
                    if (currentSocket !== null) {
                        throw new Error(`The socket already connected`);
                    }
                    forceDisconnected = false;
                    currentSocket = socket;
                    socket.getDisconnectedPromise().then((closeData) => {
                        currentSocket = null;
                        socket.drain();
                        if (forceDisconnected) {
                            socketWasConnected = false;
                            resolveDisconnectedPromises.forEach((resolve) => resolve(closeData));
                            resolveDisconnectedPromises = [];
                            messagesQueue = [];
                            messagesExpectingResponseQueue = [];
                            boundMessageListeners = [];
                        }
                    });
                    socket.bindError((error) => __awaiter(this, void 0, void 0, function* () {
                        console.error(error);
                        try {
                            yield socket.disconnect();
                        }
                        catch (_) {
                            // nothing when already disconnected
                        }
                    }));
                    reinitiateSocket();
                    if (!socketWasConnected) {
                        serverListener(wrappedSocket);
                        socketWasConnected = true;
                    }
                });
            },
        },
        listen() {
            return __awaiter(this, void 0, void 0, function* () {
                yield wsSocketWrapper.listen();
            });
        },
        close() {
            return __awaiter(this, void 0, void 0, function* () {
                yield wsSocketWrapper.close();
            });
        },
    };
}
exports.createAutoWsSocketServer = createAutoWsSocketServer;
//# sourceMappingURL=autoWsServerFactory.js.map