"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAutoReconnectingSocket = exports.createSocket = void 0;
const url_1 = require("url");
const ISocket_1 = require("../ISocket");
const events_1 = require("events");
const generator_1 = require("../../../Hash/generator");
const AutoReconnectingWSSocket_1 = require("./AutoReconnectingWSSocket");
const WebSocket = require("isomorphic-ws");
const Debug = require("debug");
const debug = Debug('@signageos/lib:WebSocket:Client:WS:createWSSocket');
const DEFAULT_PINGPONG_TIMEOUT = 30e3;
const defaultOptions = {
    enablePingPong: false,
    pingPongTimeout: DEFAULT_PINGPONG_TIMEOUT,
};
function createSocket(socketUri, onConnected, onDisconnected, onError, options = defaultOptions) {
    options = Object.assign(Object.assign({}, defaultOptions), options);
    const socketParsedUrl = (0, url_1.parse)(socketUri);
    if (socketParsedUrl.protocol === 'https:') {
        socketParsedUrl.protocol = 'wss:';
    }
    if (socketParsedUrl.protocol === 'http:') {
        socketParsedUrl.protocol = 'ws:';
    }
    const wsSocketUri = (0, url_1.format)(socketParsedUrl);
    const messageEmitter = new events_1.EventEmitter();
    const responseEmitter = new events_1.EventEmitter();
    const ws = new WebSocket(wsSocketUri);
    let isAlive = true;
    function onMessage(messageEvent) {
        debug('message', messageEvent.data);
        const message = JSON.parse(messageEvent.data);
        switch (message.type) {
            case 'response':
                responseEmitter.emit(message.responseUid);
                break;
            case 'request':
                messageEmitter.emit(message.event, message.payload);
                if (message.responseUid) {
                    const responseMessage = { type: 'response', responseUid: message.responseUid };
                    if (ws.readyState === WebSocket.OPEN) {
                        ws.send(JSON.stringify(responseMessage));
                    }
                    else {
                        debug('Socket is not open. Unconfirmed message', responseMessage, message);
                        const error = new ISocket_1.UnconfirmedMessageError(responseMessage, message, new Date());
                        onError(error);
                    }
                }
                break;
            case 'pong':
                isAlive = true;
                break;
            default:
        }
    }
    ws.addEventListener('message', onMessage);
    ws.addEventListener('error', onError);
    ws.addEventListener('open', onConnected);
    ws.addEventListener('close', onDisconnected);
    if (options.enablePingPong) {
        let pingPongInterval = null;
        ws.addEventListener('open', () => {
            pingPongInterval = setInterval(() => {
                if (!isAlive) {
                    debug('server not responding, closing websocket');
                    clearInterval(pingPongInterval);
                    ws.close(3000, 'Server is not responding');
                }
                else if (ws.readyState === WebSocket.OPEN) {
                    debug('send ping');
                    const pingMessage = { type: 'ping' };
                    ws.send(JSON.stringify(pingMessage));
                    isAlive = false;
                }
            }, options.pingPongTimeout);
        });
        ws.addEventListener('close', () => {
            if (pingPongInterval) {
                clearInterval(pingPongInterval);
            }
        });
    }
    const socket = {
        on(event, listener) {
            messageEmitter.on(event, listener);
        },
        once(event, listener) {
            messageEmitter.once(event, listener);
        },
        emit(event, payload, callback) {
            const message = { type: 'request', event, payload };
            if (callback) {
                message.responseUid = (0, generator_1.generateUniqueHash)();
                responseEmitter.once(message.responseUid, () => callback());
            }
            if (ws.readyState === WebSocket.OPEN) {
                ws.send(JSON.stringify(message));
            }
            else {
                debug('Socket is not open. Undelivered message', message);
                const error = new ISocket_1.UndeliveredEmitError(message, new Date());
                onError(error);
                throw error;
            }
        },
        removeListener(event, listener) {
            messageEmitter.removeListener(event, listener);
        },
        removeAllListeners() {
            ws.removeEventListener('message', onMessage);
            ws.removeEventListener('error', onError);
            ws.removeEventListener('open', onConnected);
            ws.removeEventListener('close', onDisconnected);
            messageEmitter.removeAllListeners();
        },
        close() {
            if (ws.readyState !== WebSocket.CLOSED && ws.readyState !== WebSocket.CLOSING) {
                ws.close(1000, 'Closed by client');
            }
            else {
                debug('Closing disconnected socket is not wanted');
            }
        },
    };
    return socket;
}
exports.createSocket = createSocket;
function createAutoReconnectingSocket(socketUri, onConnected, onDisconnected, onError, options, initialReconnectTimeout, maxReconnectTimeout) {
    const socketFactory = (onSocketConnected, onSocketDisconnected, onSocketError) => {
        return createSocket(socketUri, onSocketConnected, onSocketDisconnected, onSocketError, options);
    };
    const autoReconnectingSocket = new AutoReconnectingWSSocket_1.default(socketFactory, initialReconnectTimeout, maxReconnectTimeout);
    autoReconnectingSocket.onConnected(onConnected);
    autoReconnectingSocket.onDisconnected(onDisconnected);
    autoReconnectingSocket.onError(onError);
    return autoReconnectingSocket;
}
exports.createAutoReconnectingSocket = createAutoReconnectingSocket;
//# sourceMappingURL=createWSSocket.js.map