"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.accessLogMiddleware = void 0;
const onFinished = require("on-finished");
const auth = require("basic-auth");
function accessLogMiddleware(logCallback) {
    return (req, res, next) => {
        // record request start time and use it to calculate total time in the end
        req._startAt = process.hrtime.bigint();
        onFinished(res, (_err, resFinished) => {
            const accessLog = createAccessLogMessage(req, resFinished);
            logCallback(accessLog, req, resFinished);
        });
        next();
    };
}
exports.accessLogMiddleware = accessLogMiddleware;
/**
 * @returns A log message in format based on the Nginx combined format
 * @see https://nginx.org/en/docs/http/ngx_http_log_module.html
 */
function createAccessLogMessage(req, res) {
    var _a, _b;
    return {
        method: req.method,
        path: (_a = req.originalUrl) !== null && _a !== void 0 ? _a : req.url,
        status: res.statusCode,
        remoteAddress: getRemoteAddress(req),
        userAgent: req.get('User-Agent'),
        requestBytes: parseInt((_b = req.get('Content-Length')) !== null && _b !== void 0 ? _b : '0'),
        responseBytes: parseInt(res.get('Content-Length')),
        headers: getHeaders(req),
        referrer: getReferrer(req),
        remoteUser: getRemoteUser(req),
        tookSeconds: getTotalTime(req),
    };
}
function getRemoteAddress(req) {
    var _a, _b;
    const xForwardedFor = req.get('X-Forwarded-For');
    // if the request is coming through a proxy, the original client IP will be in the X-Forwarded-For header
    // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Forwarded-For
    if (xForwardedFor) {
        return xForwardedFor.split(',')[0].trim();
    }
    return (_b = (_a = req.ip) !== null && _a !== void 0 ? _a : req.connection.remoteAddress) !== null && _b !== void 0 ? _b : undefined;
}
function getHeaders(req) {
    const HEADERS_TO_LOG = ['cloudfront-viewer-country', 'cloudfront-viewer-asn'];
    const headers = {};
    for (const header of HEADERS_TO_LOG) {
        const value = req.get(header);
        if (value) {
            headers[header] = value;
        }
    }
    return headers;
}
function getReferrer(req) {
    var _a;
    // "Referer" is actually mis-spelled but it's the standard according to the spec
    // "Referrer" is correct spelling and sometimes used instead
    // https://en.wikipedia.org/wiki/HTTP_referer
    return (_a = req.get('Referer')) !== null && _a !== void 0 ? _a : req.get('Referrer');
}
function getRemoteUser(req) {
    var _a;
    const credentials = auth(req);
    return (_a = credentials === null || credentials === void 0 ? void 0 : credentials.name) !== null && _a !== void 0 ? _a : undefined;
}
/**
 * Returns the time between the request coming into the middleware and
 * when the response has finished being written out to the connection, in seconds.
 *
 * @return The time in seconds or 0 on error.
 */
function getTotalTime(req) {
    if (!req._startAt) {
        // missing request start time
        return 0;
    }
    // time elapsed from request start
    const ns = process.hrtime.bigint() - req._startAt;
    // convert to seconds
    return Number(ns / BigInt(1e9));
}
//# sourceMappingURL=accessLogMiddleware.js.map