"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockAcquisitionError = exports.redisLock = void 0;
const wait_1 = require("../Timer/wait");
const generator_1 = require("../Hash/generator");
const DEFAULT_RETRY_DELAY = 200;
const DEFAULT_PROLONG_INTERVAL = 10000;
const DEFAULT_TTL_EXTRA_TIMEOUT = 5000;
const redisLock = (client) => (key, options) => __awaiter(void 0, void 0, void 0, function* () {
    const acquireStartedAt = new Date();
    const retryDelay = options.retryDelay || DEFAULT_RETRY_DELAY;
    while (true) {
        const currentTime = new Date();
        if (currentTime.valueOf() - acquireStartedAt.valueOf() >= options.acquireTimeout) {
            throw new LockAcquisitionError(`Could not acquire lock on "${key}"`);
        }
        try {
            return yield createLock(client, key, options);
        }
        catch (error) {
            // Failed to set redis lock means try it again after retryDelay period
        }
        yield (0, wait_1.default)(retryDelay);
    }
});
exports.redisLock = redisLock;
class LockAcquisitionError extends Error {
}
exports.LockAcquisitionError = LockAcquisitionError;
function createLock(client, key, options) {
    return __awaiter(this, void 0, void 0, function* () {
        const lockUid = (0, generator_1.generateUniqueHash)();
        const prolongInterval = options.prolongInterval || DEFAULT_PROLONG_INTERVAL;
        const ttlExtraTimeout = options.ttlExtraTimeout || DEFAULT_TTL_EXTRA_TIMEOUT;
        const ttl = prolongInterval + ttlExtraTimeout; // add ttlExtraTimeout to redis ttl to cover network latency
        const reply = yield redisSet(client, key, lockUid, 'PX', ttl, 'NX');
        if (reply !== 'OK') {
            throw new LockAcquisitionError(`Could not create lock on "${key}". Reason: Already exists.`);
        }
        // updates redis key TTL periodically
        const prolonger = setInterval(() => __awaiter(this, void 0, void 0, function* () {
            const currentLockUid = yield redisGet(client, key);
            if (currentLockUid !== lockUid) {
                // This should never happen when used the same version of redisLock and nothing else writes to redis
                throw new Error(`Lock was stolen by other process: "${key}". Uid ${lockUid} was expected but ${currentLockUid} given.`);
            }
            yield redisSet(client, key, lockUid, 'PX', ttl, 'XX');
        }), prolongInterval);
        const releaseCb = createReleaseCallback(client, key, prolonger);
        // releases lock when timeout is reached
        const releaseTimeout = setTimeout(() => {
            console.warn(`redisLock releaseTimeout ${options.releaseTimeout}ms reached for key "${key}"`);
            releaseCb();
        }, options.releaseTimeout);
        return () => __awaiter(this, void 0, void 0, function* () {
            clearTimeout(releaseTimeout);
            yield releaseCb();
        });
    });
}
function createReleaseCallback(client, key, prolonger) {
    let runningProlonger = prolonger;
    return () => __awaiter(this, void 0, void 0, function* () {
        // interval may have been previously cleared by releaseTimeout callback
        if (runningProlonger !== null) {
            clearInterval(runningProlonger);
            runningProlonger = null;
            yield redisDel(client, key);
        }
    });
}
function redisGet(client, key) {
    return new Promise((resolve, reject) => {
        client.get(key, (error, value) => {
            if (error) {
                reject(error);
            }
            else {
                resolve(value);
            }
        });
    });
}
function redisSet(client, key, value, mode, duration, flag) {
    return new Promise((resolve, reject) => {
        client.set(key, value, mode, duration, flag, (error, reply) => {
            if (error) {
                reject(error);
            }
            else {
                resolve(reply);
            }
        });
    });
}
function redisDel(client, key) {
    return new Promise((resolve, reject) => {
        client.del(key, (error, reply) => {
            if (error) {
                reject(error);
            }
            else {
                resolve(reply);
            }
        });
    });
}
//# sourceMappingURL=redisLock.js.map