"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createRedisConnection = void 0;
const redis = require("redis");
const URL = require("url");
const genericGracefulExit_1 = require("../Events/genericGracefulExit");
const Debug = require("debug");
const genericPool = require('generic-pool');
const debug = Debug('@signageos/lib:Redis:redisConnectionFactory');
function infiniteRetryStrategy(options) {
    return Math.min(options.attempt * 100, 3e3);
}
function createRedisConnection(redisDsn, options = {}, onExit) {
    const connection = redis.createClient(redisDsn, {
        retry_strategy: options.infiniteReconnecting ? infiniteRetryStrategy : undefined,
    });
    const databases = {};
    const reconnectTimeout = 30e3;
    const gracefulExit = new genericGracefulExit_1.GenericGracefulExit('redis', reconnectTimeout, onExit);
    const pool = createPool(connection, gracefulExit, options);
    return {
        getDB(num) {
            if (databases[num] !== undefined) {
                return databases[num];
            }
            const parsedDsn = URL.parse(redisDsn, true);
            parsedDsn.query.db = num.toString();
            const subRedisDsn = URL.format(parsedDsn);
            const subConnection = createRedisConnection(subRedisDsn, options, onExit);
            subConnection.connect(); // it's practically not a promise
            databases[num] = subConnection;
            return subConnection;
        },
        connection,
        pool,
        connect() {
            connection.on('error', function (error) {
                return __awaiter(this, void 0, void 0, function* () {
                    console.error(error);
                    if (error.code === 'ECONNREFUSED' || error.code === 'EAI_AGAIN' || error.code === 'ENOTFOUND') {
                        yield gracefulExit.waitToConnectOrDie();
                    }
                });
            });
            connection.on('ready', () => {
                gracefulExit.connected();
            });
            return Promise.resolve();
        },
        close() {
            return __awaiter(this, void 0, void 0, function* () {
                yield new Promise((resolve, reject) => connection.quit((error) => (error ? reject(error) : resolve())));
                for (const num in databases) {
                    databases[num].close();
                    delete databases[num];
                }
            });
        },
        addListener() {
            // TODO implement
        },
        removeListener() {
            // TODO implement
        },
    };
}
exports.createRedisConnection = createRedisConnection;
const defaultOptions = {
    priorityRange: 3,
    min: 1,
    max: 100,
    autostart: true,
    testOnBorrow: true,
    acquireTimeoutMillis: 3e3,
};
function createPool(connection, gracefulExit, options) {
    const factory = {
        create() {
            return __awaiter(this, void 0, void 0, function* () {
                debug('Create connection');
                try {
                    const newConnection = connection.duplicate();
                    gracefulExit.connected();
                    newConnection.on('error', (error) => {
                        debug(error); // exceptions are thrown in promises
                    });
                    newConnection.on('end', () => {
                        debug('Closed connection');
                    });
                    return newConnection;
                }
                catch (e) {
                    debug('Caught error: %O', e);
                    gracefulExit.waitToConnectOrDie();
                }
                return Promise.reject('Unable to create connection redis');
            });
        },
        destroy(client) {
            return __awaiter(this, void 0, void 0, function* () {
                debug('Destroy connection');
                client.end(true);
            });
        },
        validate(client) {
            return __awaiter(this, void 0, void 0, function* () {
                return client.connected;
            });
        },
    };
    const pool = genericPool.createPool(factory, Object.assign(Object.assign({}, defaultOptions), options));
    pool.on('factoryCreateError', (error) => {
        debug('factoryCreateError %o', error);
        throw error;
    });
    pool.on('factoryDestroyError', (error) => {
        debug('factoryDestroyError %o', error);
        throw error;
    });
    return pool;
}
//# sourceMappingURL=redisConnectionFactory.js.map