"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAmqpConnection = void 0;
const amqp = require("amqplib");
const events_1 = require("events");
const ChannelProvider_1 = require("./ChannelProvider");
const QueuePublisher_1 = require("./QueuePublisher");
const QueueSubscriber_1 = require("./QueueSubscriber");
const Debug = require("debug");
const genericGracefulExit_1 = require("../Events/genericGracefulExit");
const connection_1 = require("../ThirdParty/connection");
const { createPool } = require('generic-pool');
const debug = Debug('@signageos/lib:AMQP:amqpConnectionFactory');
const defaultOptions = {
    priorityRange: 3,
    min: 1,
    max: 10,
    autostart: false,
    testOnBorrow: true,
    acquireTimeoutMillis: 3e3,
};
const defaultConnectionFactory = (dsn) => amqp.connect(dsn);
function createAmqpConnection(dsn, options = {}, onExit, 
/**
 * Option to provide own factory
 *
 * Especially useful in tests for mocking the connection
 */
connectionFactory = defaultConnectionFactory) {
    const timeout = 30e3;
    const gracefulExit = new genericGracefulExit_1.GenericGracefulExit('rabbitmq', timeout, onExit);
    const emitter = new events_1.EventEmitter();
    const factory = {
        create() {
            return __awaiter(this, void 0, void 0, function* () {
                debug('Create connection');
                try {
                    const connection = yield connectionFactory(dsn);
                    gracefulExit.connected();
                    connection.isClosed = false;
                    connection.on('error', (error) => {
                        debug(error); // exceptions are thrown in promises
                    });
                    connection.on('close', () => {
                        debug('Closed connection');
                        connection.isClosed = true;
                        emitter.emit(connection_1.ThirdPartyConnectionEvent.ConnectionClosed);
                    });
                    emitter.emit(connection_1.ThirdPartyConnectionEvent.ConnectionOpen);
                    return connection;
                }
                catch (e) {
                    debug('Caught error: %O', e);
                    gracefulExit.waitToConnectOrDie();
                }
                return Promise.reject(new Error('Unable to create connection AMQP'));
            });
        },
        destroy(connection) {
            return __awaiter(this, void 0, void 0, function* () {
                debug('Destroy connection');
                if (!connection.isClosed) {
                    yield connection.close();
                    emitter.emit(connection_1.ThirdPartyConnectionEvent.ConnectionClosed);
                }
            });
        },
        validate(connection) {
            return __awaiter(this, void 0, void 0, function* () {
                return !connection.isClosed;
            });
        },
    };
    options = Object.assign(Object.assign({}, defaultOptions), options);
    const pool = createPool(factory, options);
    pool.on('factoryCreateError', (error) => {
        debug('factoryCreateError %o', error);
        throw error;
    });
    pool.on('factoryDestroyError', (error) => {
        debug('factoryDestroyError %o', error);
        throw error;
    });
    const channelProvider = new ChannelProvider_1.default(pool);
    return {
        pool,
        channelProvider,
        queuePublisher: new QueuePublisher_1.default(),
        queueSubscriber: new QueueSubscriber_1.default(),
        connect() {
            return __awaiter(this, void 0, void 0, function* () {
                debug('connect');
                const initialConnection = yield pool.acquire();
                yield pool.release(initialConnection);
            });
        },
        close() {
            return __awaiter(this, void 0, void 0, function* () {
                debug('close');
                yield channelProvider.close();
                yield pool.drain();
                pool.clear();
            });
        },
        addListener(event, listener) {
            emitter.addListener(event, listener);
        },
        removeListener(event, listener) {
            emitter.removeListener(event, listener);
        },
    };
}
exports.createAmqpConnection = createAmqpConnection;
//# sourceMappingURL=amqpConnectionFactory.js.map