import * as winston from 'winston';
export { useRavenLogging } from './raven';
/** Central logger that aims to standardize how we work with logs across the applications.
 *
 * Goals:
 *   - Provide single API for all logs
 *   - Produce all logs in a standard format (most likely JSON) so that we can easily parse and analyze them with monitoring tools (i.e. Loki)
 *   - Provide a way to namespace logs so that we can easily filter logs by a specific part of the application
 */
export interface ILogger {
    error(message: string, meta?: object): ILogger;
    warn(message: string, meta?: object): ILogger;
    info(message: string, meta?: object): ILogger;
    debug(message: string, meta?: object): ILogger;
    /**
     * Create a new logger instance that will include the provided namespace in metadata for all logs
     *
     * Can be nested to create a hierarchy of namespaces.
     * For example:
     *
     *   ```typescript
     *   const namespacedLogger = logger.createNamespace('myNamespace');
     *   const subNamespacedLogger = namespacedLogger.createNamespace('subNamespace');
     *   subNamespacedLogger.info('This log will include both "myNamespace" and "subNamespace" in the metadata');
     *   ```
     *
     * Will produce a log with this included in the metadata:
     *
     *  ```json
     *   {
     *     "namespace": "myNamespace:subNamespace"
     *   }
     *   ```
     */
    createNamespace(namespace: string): ILogger;
}
export declare class WinstonLogger implements ILogger {
    private logger;
    private namespace?;
    constructor(logger: winston.Logger, namespace?: string | undefined);
    error(message: string, meta?: object): ILogger;
    warn(message: string, meta?: object): ILogger;
    info(message: string, meta?: object): ILogger;
    debug(message: string, meta?: object): ILogger;
    createNamespace(namespace: string): ILogger;
    private log;
}
