import { Originator } from '../Originator/Originator';
import { AlertAction } from './AlertAction';
import { AlertConditions } from './AlertCondition';
import { AlertFilter } from './AlertFilter';
import { AlertPeriodicity } from './AlertPeriodicity';
import { AlertThreshold } from './AlertThreshold';
import { AlertType } from './AlertType';
import { NotificationOptions } from './NotificationOptions';
type NullableValues<T> = {
    [P in keyof T]: T[P] | null;
};
/**
 * Widely shared properties for identification of single alert rule.
 */
export interface AlertRuleIdentifiable {
    /** Unique identifier for each alert rule across system */
    alertRuleUid: string;
}
/**
 * Creates an empty alert rule only with name and uid.
 */
export interface AlertRuleCreatable extends AlertRuleIdentifiable {
    /** User defined name for the alert rule which can be changed later */
    name: string;
    /** User creates the alert rule for the company */
    companyUid: string;
}
/**
 * Updates existing alert rule by it's uid.
 * Only specified (non undefined) values are changed. The omitted values are kept as before update.
 */
export interface AlertRuleUpdatable<T extends AlertType> extends AlertRuleIdentifiable {
    /** Type of alert which can be mapped to human readable text in UI. */
    alertType?: T;
    /** User defined name for the alert rule which can be changed later */
    name?: string;
    /** User defined description for the alert rule which can be changed later */
    description?: string;
    /**
     * Organizations which is checked against the rule.
     * There can be one or more organizations. However, some alertTypes (e.g.: POLICY_VIOLATION) can be limited only to single organization.
     */
    organizationUids?: string[];
    /**
     * Pre-filter of devices which the alert will be checked against. It's used for percentage threshold of alert creation as base.
     * The mandatory filter is organization set above. This is extended filter for example device applicationType (tizen, webos, etc.).
     * This is the first stage of 3 stages of rule before the alert is created.
     */
    filter?: AlertFilter;
    /**
     * Specify all conditions which devices has to match to state alert as applicable.
     * When device is matching the conditions, it doesn't create alert yet.
     * But the alert rule continues to be compared against specified threshold.
     * This is the second stage of 3 stages of rule before the alert is created.
     */
    conditions?: AlertConditions<T>;
    /**
     * Specify threshold of rule.
     * It is comparing number of devices matched the conditions above relatively against the total number devices filtered by filter above.
     * This is the last stage of 3 stages of rule. When this rule is matched, finally the alert is created.
     */
    threshold?: AlertThreshold;
    /**
     * Specify how often the rule will be checked against the current state of devices.
     * Also specify active windows when alert rule will be checked.
     */
    periodicity?: AlertPeriodicity;
    /**
     * Optionally specify the action of alert rule.
     * This action will happen when alert rule is creating an alert.
     * It can be omitted as null, which means that there is no action emitted when alert is created (alert is only visible in UI).
     */
    action?: AlertAction | null;
    notificationOptions?: NotificationOptions;
}
export interface IAlertRuleHistoryItem<T extends AlertType> {
    /**
     * Newly set values. Empty values were not changed.
     * Some values are always omitted to prevent data usage.
     */
    values: Partial<Omit<AlertRule<T>, 'history'>>;
    /** Date of update in history */
    updatedAt: Date;
    originator: Originator;
}
/**
 * The Alert rule entity contains all properties.
 * All properties are required except the null ones.
 * Undefined value is not used because it's reserved for no changes in update operation.
 * Single rule is a definition set which will be used for creation of specific alerts in time.
 */
export type AlertRule<T extends AlertType> = AlertRuleCreatable & NullableValues<AlertRuleUpdatable<T>> & {
    /** When the entity was created */
    createdAt: Date;
    /** All history of changes of alert rule by user. */
    history: IAlertRuleHistoryItem<T>[];
    /** Indicates whether alert rule is currently archived or not. */
    archivedAt: Date | null;
    /**
     * Indicates whether alert rule is currently paused.
     * Paused alert rules are temporarily overlooked for any checking and alert creation/update.
     */
    pausedAt: Date | null;
    /** Indicates whether and when alert rule has been checked last time. */
    lastCheckedAt: Date | null;
};
export {};
