"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BrowserClient = void 0;
const debug_1 = __importDefault(require("debug"));
const messages_1 = require("../messages");
const client_1 = require("../client");
const socket_1 = require("../socket");
const deferred_1 = require("@signageos/lib/dist/Promise/deferred");
const debug = (0, debug_1.default)('@signageos/remote-desktop:BrowserClient');
const DEFAULT_WS_PING_PONG_TIMEOUT_MS = 30 * 1000;
class BrowserClient {
    constructor(
    /** The unique identifier of device (not DUID) */
    deviceUid, screenshotRenderer) {
        this.deviceUid = deviceUid;
        this.screenshotRenderer = screenshotRenderer;
        this.deviceConnected = false;
    }
    start(options) {
        return __awaiter(this, void 0, void 0, function* () {
            debug('Starting BrowserClient', options);
            if (this.socket) {
                throw new Error('Already started');
            }
            yield this.connectWebSocket(options);
            debug('Connected to WebSocket');
            const { connectedPromise } = this.bindSocketEvents(options);
            return (0, client_1.createStartedClient)(this, connectedPromise, options);
        });
    }
    stop() {
        return __awaiter(this, void 0, void 0, function* () {
            var _a;
            debug('Stopping BrowserClient');
            if (!this.socket) {
                throw new Error('Not started');
            }
            (_a = this.socket) === null || _a === void 0 ? void 0 : _a.close();
            this.socket = undefined;
            this.deviceConnected = false;
        });
    }
    isStarted() {
        return Boolean(this.socket);
    }
    isDeviceConnected() {
        return this.deviceConnected;
    }
    sendMouseEvent(payload) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.sendEvent(messages_1.Message.MouseEvent, payload);
        });
    }
    sendKeyEvent(payload) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.sendEvent(messages_1.Message.KeyEvent, payload);
        });
    }
    sendButtonEvent(payload) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.sendEvent(messages_1.Message.ButtonEvent, payload);
        });
    }
    connectWebSocket(options) {
        return __awaiter(this, void 0, void 0, function* () {
            const xAuth = `${options.auth.tokenId}:${options.auth.token}`;
            const serverUri = `${options.serverUrl}/browser?deviceUid=${this.deviceUid}&x-auth=${xAuth}`;
            const webSocketOptions = this.createWebSocketOptions(options);
            this.socket = yield (0, socket_1.connectSocket)(serverUri, webSocketOptions);
        });
    }
    createWebSocketOptions(options) {
        var _a;
        return {
            enablePingPong: true,
            pingPongTimeout: (_a = options.wsPingPongTimeoutMs) !== null && _a !== void 0 ? _a : DEFAULT_WS_PING_PONG_TIMEOUT_MS,
        };
    }
    bindSocketEvents(options) {
        if (!this.socket) {
            throw new Error('Socket not initialized');
        }
        const connectedDeferred = (0, deferred_1.createDeferred)();
        this.socket.on(messages_1.Message.ScreenshotRecorded, (payload) => __awaiter(this, void 0, void 0, function* () {
            debug('ScreenshotRecorded', payload);
            yield this.screenshotRenderer.render(payload);
        }));
        this.socket.on(messages_1.Message.DeviceConnected, () => {
            var _a;
            debug('DeviceConnected');
            this.deviceConnected = true;
            (_a = options.onStatus) === null || _a === void 0 ? void 0 : _a.call(options, true);
            connectedDeferred.resolve();
        });
        this.socket.on(messages_1.Message.DeviceDisconnected, () => {
            var _a;
            debug('DeviceDisconnected');
            this.deviceConnected = false;
            (_a = options.onStatus) === null || _a === void 0 ? void 0 : _a.call(options, false);
        });
        return {
            connectedPromise: connectedDeferred.promise,
        };
    }
    sendEvent(message, payload) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.socket) {
                throw new Error('Socket not connect. Use start() method to connect to the server first.');
            }
            if (!this.deviceConnected) {
                throw new Error('Device not connected');
            }
            debug('Sending event', message, payload);
            this.socket.emit(message, payload);
        });
    }
}
exports.BrowserClient = BrowserClient;
//# sourceMappingURL=BrowserClient.js.map