const VERSION_HEADER = '## ';
const CHANGES_HEADER = '### ';
const LOG_LINE = '- ';

exports.getPipedInput = function () {
	return new Promise((resolve, reject) => {
		let inputData;
		process.stdin.resume();
		process.stdin.setEncoding('utf8');
		process.stdin.on('data', (chunk) => (inputData += chunk));
		process.stdin.on('end', () => resolve(inputData));
		process.stdin.on('error', (error) => reject(error));
	});
};

exports.getGroupedChangeLog = function (changeLog) {
	const newLines = changeLog
		.split(/\n|\r/)
		.filter((line) => line.startsWith(VERSION_HEADER) || line.startsWith(CHANGES_HEADER) || line.startsWith(LOG_LINE));
	const { groupedLog } = newLines.reduce(
		(reduction, line) => {
			if (line.startsWith(VERSION_HEADER)) {
				const version = line
					.substring(VERSION_HEADER.length)
					.match(/^\[?(([\d\w\.\-\+])+)\]?/)[1]
					.trim();
				if (reduction.groupedLog[version] !== undefined) {
					throw new Error(`The version "${version}" is duplicated in CHANGELOG`);
				}
				return {
					...reduction,
					lastVersion: version,
					groupedLog: {
						...reduction.groupedLog,
						[version]: reduction.groupedLog[version] || {},
					},
				};
			} else if (line.startsWith(CHANGES_HEADER)) {
				const type = line.substring(CHANGES_HEADER.length).trim();
				if (reduction.groupedLog[reduction.lastVersion] === undefined) {
					throw new Error(`The type "${type}" is not under any version section in CHANGELOG`);
				}
				if (reduction.groupedLog[reduction.lastVersion][type] !== undefined) {
					throw new Error(`The type "${type}" is duplicated at version "${reduction.lastVersion}" in CHANGELOG`);
				}
				return {
					...reduction,
					lastType: type,
					groupedLog: {
						...reduction.groupedLog,
						[reduction.lastVersion]: {
							...reduction.groupedLog[reduction.lastVersion],
							[type]: reduction.groupedLog[reduction.lastVersion][type] || [],
						},
					},
				};
			} else {
				return {
					...reduction,
					groupedLog: {
						...reduction.groupedLog,
						[reduction.lastVersion]: {
							...reduction.groupedLog[reduction.lastVersion],
							[reduction.lastType]: [...reduction.groupedLog[reduction.lastVersion][reduction.lastType], line.substring(LOG_LINE.length)],
						},
					},
				};
			}
		},
		{ groupedLog: {}, lastType: null, lastVersion: null },
	);
	return groupedLog;
};
