import { z } from 'zod';
import { fs } from 'zx';

const nodeEnvSchema = z.union([z.literal('development'), z.literal('test'), z.literal('production')]);

const emulatorConfigSchema = z.intersection(
	// Base config
	z.object({
		NODE_ENV: nodeEnvSchema,
		// `environment is infered from `NODE_ENV`
		environment: nodeEnvSchema,
		PACKAGES_PATH: z.string().optional(),
		PORT: z.string().regex(/^\d+$/).optional(),

		base_url: z.string().url(),
		socket_uri: z.string().url(),
		static_base_url: z.string().url(),
		upload_base_url: z.string().url(),
		weinre_server_url: z.string().url(),
		synchronizer_server_url: z.string().url(),
		ping_url: z.string().url(),

		http_auth_public_key: z.string(),
		check_interval_ms: z.string().regex(/^\d+$/),

		auto_verification_organization_uid: z.string().optional(),
		subscription_type: z.string().optional(),
	}),
	z.intersection(
		// Optional raven config
		z.union([
			z.object({
				raven_enabled: z.literal('1'),
				raven_dsn: z.string().url(),
			}),
			z.object({
				raven_enabled: z.literal('0').optional(),
			}),
		]),
		// Optional bundled applet config
		z.union([
			z.object({
				bundled_applet: z.literal('1'),
				bundled_applet_uid: z.string(),
				bundled_applet_version: z.string(),
				bundled_applet_checksum: z.string(),
				bundled_applet_binary_file: z.string(),
				bundled_applet_front_applet_version: z.string(),
				bundled_applet_front_applet_binary_file: z.string(),
				bundled_applet_config: z.string(),
			}),
			z.object({
				bundled_applet: z.literal('0').optional(),
			}),
		]),
	),
);

/** @param {string} absoluteRootPath */
export const getEmulatorConfig = async (absoluteRootPath) => {
	const environment = process.env.NODE_ENV ?? 'development';

	let localEnv = await fs.readJSON(absoluteRootPath + '/config/env.' + environment + '.json').catch(() => ({}));

	const v = {
		environment,
		...process.env,
		...localEnv,
	};

	const config = emulatorConfigSchema.safeParse({
		...v,
		NODE_ENV: v.environment === 'dev' ? 'development' : v.NODE_ENV,
	});

	if (!config.success) {
		console.log(config.error.format());
		process.exit(1);
	}

	return config.data;
};
