import { Bundler, stopwatch } from '@signageos/lib-bundler';
import { context } from 'esbuild';
import { fs } from 'zx';
import { buildOptions, parameters } from '../bundler.config.mjs';
import { CABundler } from './CABundler.mjs';

const bundler = new Bundler({
	outdir: './dist',
	parameters,
	argv: process.argv,
	serveOptions: {
		port: 8080,
	},
});

await Bundler.prepareEs6Folder();

await stopwatch('Building the Front Display', [
	stopwatch('es6', async () => {
		await bundler.compileTypescript();
		await Bundler.copyAssetsToEs6Folder();
	}),
	async () => {
		await stopwatch('Building polyfill', [CABundler.buildPolyfill({ dest: './dist/polyfill.js' })]);
		await stopwatch('Bundling', async () => {
			await bundler.writeIndexFile('./assets/index.html');
			const ctx = await context({
				...bundler.esbuildOptions,
				...buildOptions,
			});
			await bundler.build(ctx);
		});

		await stopwatch('Transpiling to ES5', [
			bundler.transpileToES5({
				minify: !process.env.NO_MINIFY,
				env: {
					targets: {
						chrome: '28',
					},
				},
			}),
		]);

		await stopwatch('Move files', async () => {
			const filesToCopy = [
				{ src: './node_modules/@signageos/front-osd/dist', dest: './dist/osd' },
				{ src: './node_modules/weinre/web/target/target-script.js', dest: `./dist/weinre.js` },
				{ src: './dist/bundle.css', dest: './dist/styles.css' },
				{ src: './node_modules/zip.js/WebContent/z-worker.js', dest: './dist/z-worker.js' },
				{ src: './node_modules/zip.js/WebContent/inflate.js', dest: './dist/inflate.js' },
				{ src: './node_modules/zip.js/WebContent/deflate.js', dest: './dist/deflate.js' },
				{ src: './assets/console.js', dest: './dist/console.js' },
			];

			for (const { src, dest } of filesToCopy) {
				if (await fs.pathExists(src)) {
					await fs.copy(src, dest, { overwrite: true });
				} else {
					console.log(`The file ${src} does not exist.`);
				}
			}
		});
	},
]);
