import { Bundler } from '@signageos/lib-bundler';
import compat from 'core-js-compat';
import { build } from 'esbuild';
import { networkInterfaces } from 'os';
import { chalk, fs } from 'zx';

export class CABundler extends Bundler {
	/**
	 * @param {{
	 *   entries?: string[];
	 *   targets?: NonNullable<Parameters<typeof compat>[0]>["targets"] & object
	 *   dest?: string
	 * }} opts
	 */
	static async buildPolyfill({ entries = [], targets, dest }) {
		try {
			console.info(`Building polyfill`);
			const { list } = /** @type {{ list: string[] }} */ (
				compat({
					modules: ['core-js/actual'],
					targets: {
						chrome: 28,
						safari: 5, // Old webkit devices correspond to safari 5,
						...targets,
					},
				})
			);

			const polyfills = createSourceFile(['./src/polyfills/index.ts', ...entries, ...list.map(prependCoreJsPath)]);

			await build({
				bundle: true,
				stdin: {
					contents: polyfills,
					resolveDir: process.cwd(),
				},
				outfile: dest,
			});
		} catch (error) {
			console.error(`Failed to build polyfill: ${error}`);
			throw error;
		}
	}

	/**
	 * @param {string} dest
	 */
	static async copyFrontDisplayAssets(dest) {
		await bulkCopy([
			{ src: './node_modules/@signageos/front-osd/dist', dest: `${dest}/osd` },
			{ src: './node_modules/@signageos/front-display/dist/polyfill.js', dest: `${dest}/polyfill.js` },
			{ src: './node_modules/@signageos/front-display/dist/console.js', dest: `${dest}/console.js` },
			{ src: './node_modules/@signageos/front-display/dist/webWorker.js', dest: `${dest}/webWorker.js` },
			{ src: './node_modules/weinre/web/target/target-script.js', dest: `${dest}/weinre.js` },
			{ src: './node_modules/zip.js/WebContent/z-worker.js', dest: `${dest}/z-worker.js` },
			{ src: './node_modules/zip.js/WebContent/inflate.js', dest: `${dest}/inflate.js` },
			{ src: './node_modules/zip.js/WebContent/deflate.js', dest: `${dest}/deflate.js` },
		]);
	}

	/**
	 * Sets up valid BUILDTIME and NODE_ENV env variables
	 * @param {string[]} argv
	 */
	static ensureValidEnv(argv) {
		process.env.BUILDTIME = '1';

		process.env.NODE_ENV ||= argv.includes('--prod') ? 'production' : 'development';

		// Backward compatibility for the old NODE_ENV values
		if (process.env.NODE_ENV === 'dev') {
			console.warn(chalk.yellow(`Invalid NODE_ENV "${process.env.NODE_ENV}", using "development" instead.`));
			process.env.NODE_ENV = 'development';
		}

		if (process.env.NODE_ENV === 'prod') {
			console.warn(chalk.yellow(`Invalid NODE_ENV "${process.env.NODE_ENV}", using "production" instead.`));
			process.env.NODE_ENV = 'production';
		}

		if (!['production', 'development', 'test'].includes(process.env.NODE_ENV)) {
			console.warn(chalk.yellow(`Using unusual NODE_ENV value "${process.env.NODE_ENV}".`));
		}
	}

	static getLocalIP() {
		const localIp = Object.values(networkInterfaces())
			.flat()
			.find((i) => i?.family === 'IPv4' && !i.internal);

		if (!localIp) {
			throw new Error('No valid local IP found');
		}
		return localIp.address;
	}
}

/** @param {string[]} list */
const createSourceFile = (list) => list.map(wrapEntryByRequireCall).join(';\n');

/** @param {string} element */
const prependCoreJsPath = (element) => `core-js/modules/${element}`;

/** @param {string} module */
const wrapEntryByRequireCall = (module) => `require('${module}')`;

/**
 * @param {{ src: string, dest: string }[]} instructions
 */
export const bulkCopy = async (instructions) => {
	for (const { src, dest } of instructions) {
		if (await fs.pathExists(src)) {
			await fs.copy(src, dest, { overwrite: true });
		} else {
			console.error(`The file ${src} does not exist.`);
		}
	}
};
