import WebWorkerType from './WebWorker/WebWorkerType';
import { createAppletTimingControllerWebWorker } from './WebWorker/webWorkerFactory';
import { ShowNextAppletTiming } from './Front/AppletTiming/appletTimingActions';
import IWebWorker from './WebWorker/IWebWorker';
import { ISupportedConsoleLevel, supportedConsoleLevels } from './WebWorker/webWorkerConsole';
import { IParentMessageData } from './WebWorker/parentMessages';
import { IWebWorkerMessageData } from './WebWorker/webWorkerMessages';

let webWorker: IWebWorker<any, any>;

const postMessageToParent = (data: IParentMessageData) => {
	return postMessage(data); // uses different interface of postMessage in webworker environment (IDE does not check it correctly)
};

function createWebWorker(type: WebWorkerType) {
	// Monkey patch global console object
	console = supportedConsoleLevels.reduce(
		(overwritten: Console, level: ISupportedConsoleLevel) => ({
			...overwritten,
			[level]: (...args: any[]) => postMessageToParent({ name: 'console', level, args }),
		}),
		console ?? ({} as Console),
	);

	switch (type) {
		case WebWorkerType.APPLET_TIMING_CONTROLLER:
			return createAppletTimingControllerWebWorker((message: ShowNextAppletTiming) => {
				postMessageToParent({ name: 'message', message });
			});
		default:
			throw new Error('Invalid type: ' + type);
	}
}

addEventListener('message', async (event: MessageEvent<IWebWorkerMessageData>) => {
	try {
		switch (event.data.name) {
			case 'start':
				webWorker = createWebWorker(event.data.type);
				await webWorker.start(event.data.initData);
				break;
			case 'message':
				const message = event.data.message;
				if (webWorker) {
					webWorker.onMessage(message);
				} else {
					throw new Error("Message received but web worker haven't been initialized yet: " + message);
				}
				break;
			default:
				throw new Error('invalid message name: ' + (event as any).data.name);
		}
	} catch (error) {
		postMessageToParent({
			name: 'error',
			errorMessage: error.message,
			errorName: error.name,
			errorStack: error.stack,
		});
		throw error;
	}
});
