import { EventEmitter } from 'events';
import IMasterWebWorker from './IMasterWebWorker';
import WebWorkerType from './WebWorkerType';
import { ILoopModel } from '../Front/AppletTiming/appletTimingControllerSagas';
import { ShowNextAppletTiming } from '../Front/AppletTiming/appletTimingActions';
import IWebWorker from './IWebWorker';
import { createAppletTimingControllerWebWorker } from './webWorkerFactory';
import { IWebWorkerMessageData } from './webWorkerMessages';
import { IParentMessageData } from './parentMessages';

const WEB_WORKER_FILE_PATH = 'webWorker.js';

export interface IWebWorkerFactory {
	createAppletTimingController(): IMasterWebWorker<ILoopModel, ShowNextAppletTiming, undefined>;
}

export function createWebWorkerFactory(appVersion: string, createWebWorkerInstance?: () => Worker): IWebWorkerFactory {
	return {
		createAppletTimingController: () => {
			return createWebWorker<ILoopModel, ShowNextAppletTiming, undefined>(
				WebWorkerType.APPLET_TIMING_CONTROLLER,
				appVersion,
				createWebWorkerInstance,
			);
		},
	};
}

export function createSameThreadWebWorkerFactory(): IWebWorkerFactory {
	return {
		createAppletTimingController: () => {
			return createSameThreadWebWorker<ILoopModel, ShowNextAppletTiming, undefined>(
				(postMessage: (message: ShowNextAppletTiming) => void) => createAppletTimingControllerWebWorker(postMessage),
			);
		},
	};
}

function createWebWorker<TInputMessage, TOutputMessage, TInitData>(
	type: WebWorkerType,
	appVersion: string,
	createWebWorkerInstance?: () => Worker,
): IMasterWebWorker<TInputMessage, TOutputMessage, TInitData> {
	// TODO make fallback for platforms that dont support web workers
	const webWorkerPath = WEB_WORKER_FILE_PATH + '?v=' + appVersion;
	const webWorker = createWebWorkerInstance ? createWebWorkerInstance() : new Worker(webWorkerPath);
	return {
		start: (initData: TInitData) => webWorker.postMessage(createStartMessage<TInitData>(type, initData)),
		postMessage: (message: TInputMessage) => webWorker.postMessage(createMessageMessage<TInputMessage>(message)),
		onMessage: (callback: (message: TOutputMessage) => void) => {
			webWorker.addEventListener('message', (event: MessageEvent<IParentMessageData>) => {
				switch (event.data.name) {
					case 'console':
						console[event.data.level](...event.data.args);
						break;
					case 'error':
						const error = new Error(event.data.errorMessage);
						error.name = event.data.errorName;
						error.stack = event.data.errorStack;
						throw error;
					case 'message':
						callback(event.data.message);
						break;
					default:
						throw new Error('invalid message name: ' + (event as any).data.name);
				}
			});
		},
	};
}

function createMessageMessage<TInputMessage>(message: TInputMessage): IWebWorkerMessageData {
	return { name: 'message', message };
}

function createStartMessage<TInitData>(type: WebWorkerType, initData: TInitData): IWebWorkerMessageData {
	return { name: 'start', type, initData };
}

function createSameThreadWebWorker<TInputMessage, TOutputMessage, TInitData>(
	createWebWorkerLocal: (postMessage: (message: TOutputMessage) => void) => IWebWorker<TInputMessage, TInitData>,
): IMasterWebWorker<TInputMessage, TOutputMessage, TInitData> {
	const eventEmitter = new EventEmitter();
	const webWorker = createWebWorkerLocal((message: TOutputMessage) => eventEmitter.emit('message', message));
	return {
		start: (initData: TInitData) => webWorker.start(initData),
		postMessage: (message: TInputMessage) => webWorker.onMessage(message),
		onMessage: (callback: (message: TOutputMessage) => void) => {
			eventEmitter.addListener('message', callback);
		},
	};
}
