import { IPrepareOptions } from './IVideoPlayer';
import IVideo from './IVideo';
import { VideoEventType } from './IVideoEvent';

enum VideoState {
	IDLE = 0,
	PREPARED = 1,
	PLAYING = 2,
}

interface IVideoArguments {
	uri: string;
	x: number;
	y: number;
	width: number;
	height: number;
}

/**
 * Wrapper that enables video instance to have state
 */
export default class VideoWithState<TOptions extends IPrepareOptions> implements IVideo<TOptions> {
	private state: VideoState = VideoState.IDLE;
	private args: IVideoArguments | null = null;
	private resolveOnceIdleCallbacks: (() => void)[] = [];

	constructor(private video: IVideo) {}

	public async prepare(uri: string, x: number, y: number, width: number, height: number, options?: TOptions): Promise<void> {
		await this.video.prepare(uri, x, y, width, height, options);
		this.state = VideoState.PREPARED;
		this.args = { uri, x, y, width, height };
	}

	public async play(): Promise<void> {
		await this.video.play();
		this.state = VideoState.PLAYING;
	}

	public pause(): Promise<void> {
		return this.video.pause();
	}

	public resume(): Promise<void> {
		return this.video.resume();
	}

	public async stop(): Promise<void> {
		await this.video.stop();
		this.state = VideoState.IDLE;
		this.args = null;
		this.resolveWaitUntilIdlePromises();
	}

	public waitUntilIdle(): Promise<void> {
		return new Promise<void>((resolve: () => void) => {
			this.resolveOnceIdleCallbacks.push(resolve);
		});
	}

	public addEventListener(eventName: VideoEventType, listener: () => void): void {
		this.video.addEventListener(eventName, listener);
	}

	public removeAllEventListeners(): void {
		this.video.removeAllEventListeners();
	}

	public isIdle() {
		return this.state === VideoState.IDLE;
	}

	public isPrepared() {
		return this.state === VideoState.PREPARED;
	}

	public getArguments() {
		return this.args;
	}

	public getDuration(): number {
		return this.video.getDuration();
	}

	private resolveWaitUntilIdlePromises() {
		for (let i = 0; i < this.resolveOnceIdleCallbacks.length; i++) {
			this.resolveOnceIdleCallbacks[i]();
		}
		this.resolveOnceIdleCallbacks = [];
	}
}
