import IVideoPlayer, { IPrepareOptions } from './IVideoPlayer';
import IVideo from './IVideo';
import { locked } from '../Lock/lockedDecorator';
import IVideoEventEmitter from './IVideoEventEmitter';
import { EventEmitter } from 'events';
import IVideoEvent from './IVideoEvent';
import { NoMoreAvailableVideosError } from '../NativeDevice/Error/videoErrors';
import { debug } from '@signageos/lib/dist/Debug/debugDecorator';
import VideoWithState from './VideoWithState';
import VideoOptions from './VideoOptions';

const LOCK_KEY = 'ProprietaryVideoPlayer.video';
const DEBUG_NAMESPACE = '@signageos/front-display:Video:ProprietaryVideoPlayer';

/**
 * Generic video player
 *
 * Since the state management of videos is the same on almost all the platforms,
 * it makes sense to have this class that implements the state management but doesn't care about the video implemention itself.
 * This class can be used directly or it can be wrapped in another class via composition that would extend it's capabilities.
 */
export default class ProprietaryVideoPlayer<TOptions extends IPrepareOptions = IPrepareOptions> implements IVideoPlayer<TOptions> {
	private videos: VideoWithState<TOptions>[];

	private videoOptions: VideoOptions<TOptions>;

	constructor(videos: IVideo[]) {
		this.videos = videos.map((video: IVideo) => new VideoWithState(video));
		this.videoOptions = new VideoOptions<TOptions>();
	}

	public getMaxVideoCount(): number {
		return this.videos.length;
	}

	@locked(LOCK_KEY)
	@debug(DEBUG_NAMESPACE)
	public async prepare(uri: string, x: number, y: number, width: number, height: number, options?: TOptions): Promise<void> {
		if (typeof options === 'object') {
			this.videoOptions.setOptions(uri, x, y, width, height, options);
		}

		await this.prepareVideoIfNotPrepared(uri, x, y, width, height);
	}

	@locked(LOCK_KEY)
	@debug(DEBUG_NAMESPACE)
	public async play(uri: string, x: number, y: number, width: number, height: number): Promise<IVideoEventEmitter> {
		try {
			return await this.playVideo(uri, x, y, width, height);
		} catch (error) {
			if (error instanceof NoMoreAvailableVideosError) {
				return this.playVideoOnceAvailable(uri, x, y, width, height);
			} else {
				throw error;
			}
		}
	}

	@locked(LOCK_KEY)
	@debug(DEBUG_NAMESPACE)
	public async stop(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		let playingVideo: VideoWithState<TOptions>;
		try {
			playingVideo = this.getVideoByArgumentsOrThrowException(uri, x, y, width, height);
		} catch (error) {
			console.warn("Attempt to stop video that's not playing");
			return;
		}

		await playingVideo.stop();
		this.videoOptions.clearOptions(uri, x, y, width, height);
		playingVideo.removeAllEventListeners();
	}

	@locked(LOCK_KEY)
	@debug(DEBUG_NAMESPACE)
	public async pause(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		const playingVideo = this.getVideoByArgumentsOrThrowException(uri, x, y, width, height);
		await playingVideo.pause();
	}

	@locked(LOCK_KEY)
	@debug(DEBUG_NAMESPACE)
	public async resume(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		const playingVideo = this.getVideoByArgumentsOrThrowException(uri, x, y, width, height);
		await playingVideo.resume();
	}

	@locked(LOCK_KEY)
	@debug(DEBUG_NAMESPACE)
	public async clearAll(): Promise<void> {
		await Promise.all(
			this.videos.map(async (video: VideoWithState<TOptions>) => {
				try {
					await video.stop();
					const args = video.getArguments();
					if (args) {
						this.videoOptions.clearOptions(args?.uri, args?.x, args?.y, args?.width, args?.height);
					}
				} catch (error) {
					console.warn('failed to clear video', error);
				}
			}),
		);
	}

	@locked(LOCK_KEY)
	@debug(DEBUG_NAMESPACE)
	public async getDuration(uri: string, x: number, y: number, width: number, height: number): Promise<number> {
		const video = this.getIdleVideoOrThrowException();
		try {
			await this.prepareVideo(video, uri, x, y, width, height);
			const duration = video.getDuration();
			return duration;
		} finally {
			await video.stop();
		}
	}

	private async playVideo(uri: string, x: number, y: number, width: number, height: number): Promise<IVideoEventEmitter> {
		const video: VideoWithState<TOptions> = await this.prepareVideoIfNotPrepared(uri, x, y, width, height);
		await video.play();
		return this.createVideoEventEmitter(video);
	}

	private async prepareVideoIfNotPrepared(uri: string, x: number, y: number, width: number, height: number) {
		let video: VideoWithState<TOptions>;
		try {
			video = this.getVideoByArgumentsOrThrowException(uri, x, y, width, height);
		} catch (error) {
			video = this.getIdleVideoOrThrowException();
			await this.prepareVideo(video, uri, x, y, width, height);
		}
		return video;
	}

	private async prepareVideo(video: IVideo, uri: string, x: number, y: number, width: number, height: number) {
		const options = this.videoOptions.getOptions(uri, x, y, width, height);
		await video.prepare(uri, x, y, width, height, options);
		return video;
	}

	private playVideoOnceAvailable(uri: string, x: number, y: number, width: number, height: number): IVideoEventEmitter {
		const videoEventEmitter = new EventEmitter();
		this.waitUntilSomeVideoBecomesIdle().then(async () => {
			const video = await this.play(uri, x, y, width, height);
			video.addListener('playing', (event: IVideoEvent) => videoEventEmitter.emit('playing', event));
			video.addListener('ended', (event: IVideoEvent) => videoEventEmitter.emit('ended', event));
			video.addListener('error', (event: IVideoEvent) => videoEventEmitter.emit('error', event));
			video.addListener('stopped', (event: IVideoEvent) => videoEventEmitter.emit('stopped', event));
		});
		return videoEventEmitter;
	}

	private waitUntilSomeVideoBecomesIdle() {
		return Promise.race(this.videos.map((video: VideoWithState<TOptions>) => video.waitUntilIdle()));
	}

	private getIdleVideoOrThrowException() {
		for (let video of this.videos) {
			if (video.isIdle()) {
				return video;
			}
		}

		// if no idle videos available, get a prepared video instead
		for (let video of this.videos) {
			if (video.isPrepared()) {
				return video;
			}
		}

		throw new NoMoreAvailableVideosError();
	}

	private getVideoByArgumentsOrThrowException(uri: string, x: number, y: number, width: number, height: number) {
		for (let video of this.videos) {
			const videoArguments = video.getArguments();
			if (
				videoArguments &&
				videoArguments.uri === uri &&
				videoArguments.x === x &&
				videoArguments.y === y &&
				videoArguments.width === width &&
				videoArguments.height === height
			) {
				return video;
			}
		}

		throw new Error('Video with arguments ' + JSON.stringify({ uri, x, y, width, height }) + ' not found');
	}

	private createVideoEventEmitter(video: VideoWithState<TOptions>): IVideoEventEmitter {
		const videoEventEmitter = new EventEmitter();
		const videoEvent = {
			srcArguments: video.getArguments(),
		};
		video.addEventListener('playing', () => {
			videoEventEmitter.emit('playing', { type: 'playing', ...videoEvent });
		});
		video.addEventListener('ended', () => {
			videoEventEmitter.emit('ended', { type: 'ended', ...videoEvent });
		});
		video.addEventListener('error', () => {
			videoEventEmitter.emit('error', { type: 'error', ...videoEvent });
		});
		video.addEventListener('stopped', () => {
			videoEventEmitter.emit('stopped', { type: 'stopped', ...videoEvent });
		});
		return videoEventEmitter;
	}
}
