import { EventEmitter } from 'events';
import Debug from 'debug';
import { debug } from '@signageos/lib/dist/Debug/debugDecorator';
import { IPrepareOptions } from './IVideoPlayer';
import IVideo from './IVideo';
import { Cancelable, debounce } from 'lodash';

const DEBUG_NAMESPACE = '@signageos/front-display:Video:HTMLVideo';
const logDebug = Debug(DEBUG_NAMESPACE);

const STREAM_RECONNECT_TIMEOUT = 2e3;

export default class HTMLVideo implements IVideo {
	private videoWrapper: HTMLDivElement;
	private video: HTMLVideoElement;
	private currentOptions: IPrepareOptions | undefined;
	private eventEmitter: EventEmitter;
	private endStreamWhenCannotReconnectDebounce: (() => void) & Cancelable = debounce(
		() => this.eventEmitter.emit('ended'),
		STREAM_RECONNECT_TIMEOUT,
	);

	constructor(
		private window: Window,
		private foregroundWrapperElement: HTMLElement = window.document.body,
		private backgroundWrapperElement?: HTMLElement,
	) {
		this.eventEmitter = new EventEmitter();
		this.initializeVideo();
		this.resetEventEmitter();
	}

	@debug(DEBUG_NAMESPACE)
	public async play() {
		try {
			await this.playVideo();
		} catch (error) {
			logDebug('playVideo error', error);
			this.ifChromeLogWarningWithLinkToDocs();
			await this.playVideoMutedFallback();
		}
		this.show();
	}

	@debug(DEBUG_NAMESPACE)
	public async stop() {
		// https://html.spec.whatwg.org/multipage/media.html#best-practices-for-authors-using-media-elements
		this.clearVideoTagStyle();
		this.video.pause();
		this.video.removeAttribute('src');
		this.video.load();
		this.eventEmitter.emit('stopped');
	}

	@debug(DEBUG_NAMESPACE)
	public async pause() {
		this.video.pause();
	}

	@debug(DEBUG_NAMESPACE)
	public async resume(): Promise<void> {
		await this.video.play();
	}

	@debug(DEBUG_NAMESPACE)
	public prepare(uri: string, x: number, y: number, width: number, height: number, options: IPrepareOptions = {}) {
		this.currentOptions = options;
		return new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			this.setupVideoObjectStyle(x, y, width, height);

			if (typeof options.volume !== 'undefined') {
				this.video.volume = options.volume / 100;
			} else {
				this.video.volume = 1;
			}

			this.prepareSequenceListeners()
				.then(() => {
					this.endIfCannotRecover();
					resolve();
				})
				.catch((e: Error) => reject(e));
			// Wait one tick because prepareSequenceListeners is in new Promise which delays by 1 tick as well
			setTimeout(() => this.setVideoSourceAndLoad(uri));
		});
	}

	public addEventListener(eventName: 'playing' | 'ended' | 'stopped' | 'error', listener: () => void): void {
		this.eventEmitter.addListener(eventName, listener);
	}

	public removeAllEventListeners(): void {
		this.resetEventEmitter();
	}

	@debug(DEBUG_NAMESPACE)
	public setVolume(volumePercentage: number) {
		this.video.volume = volumePercentage / 100;
	}

	public getDuration() {
		return this.video.duration * 1e3;
	}

	private async playVideo() {
		this.video.muted = false;
		this.video.currentTime = 0;
		await this.video.play();
	}

	private async playVideoMutedFallback() {
		this.video.muted = true;
		this.video.currentTime = 0;
		await this.video.play();
	}

	/**
	 * This method loads video wrapped by this HTML video class.
	 *
	 * Set src and load has to be called in one tick otherwise our listeners logic is broken
	 * by browser auto load behavior which we can not change.
	 *
	 * More on https://developer.mozilla.org/en-US/docs/Web/API/HTMLMediaElement/loadstart_event
	 * and here https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-preload.
	 */
	private setVideoSourceAndLoad(uri: string) {
		this.video.setAttribute('src', uri);
		this.video.load();
	}

	private hide() {
		this.videoWrapper.style.visibility = 'hidden';
		if (this.videoWrapper.parentElement !== this.foregroundWrapperElement) {
			this.foregroundWrapperElement.appendChild(this.videoWrapper);
		}
	}

	private show() {
		if (this.currentOptions && this.currentOptions.background && this.backgroundWrapperElement) {
			this.backgroundWrapperElement.appendChild(this.videoWrapper);
		}
		this.videoWrapper.style.visibility = 'visible';
	}

	private clearVideoTagStyle() {
		this.videoWrapper.style.left = '0px';
		this.videoWrapper.style.top = '0px';
		this.videoWrapper.style.width = '0px';
		this.videoWrapper.style.height = '0px';
		this.hide();
	}

	private setupVideoObjectStyle(x: number, y: number, width: number, height: number) {
		this.videoWrapper.setAttribute('width', width.toString());
		this.videoWrapper.setAttribute('height', height.toString());
		this.videoWrapper.style.left = x + 'px';
		this.videoWrapper.style.top = y + 'px';
		this.videoWrapper.style.width = width + 'px';
		this.videoWrapper.style.height = height + 'px';
	}

	private initializeVideo() {
		this.videoWrapper = this.window.document.createElement('div');
		this.videoWrapper.style.background = '#000000';
		this.videoWrapper.style.position = 'absolute';
		this.hide();
		this.foregroundWrapperElement.appendChild(this.videoWrapper);
		this.video = this.window.document.createElement('video');
		this.video.setAttribute('class', 'default-video');
		this.video.setAttribute('width', '100%');
		this.video.setAttribute('height', '100%');
		this.video.style.width = '100%';
		this.videoWrapper.appendChild(this.video);
		this.video.addEventListener('ended', () => {
			this.eventEmitter.emit('ended');
		});
		this.video.addEventListener('error', () => {
			this.eventEmitter.emit('error');
		});
		this.debugVideo();
	}

	private endIfCannotRecover = () => {
		this.video.addEventListener('stalled', this.endStreamWhenCannotReconnectDebounce);
		this.video.addEventListener('error', this.endStreamWhenCannotReconnectDebounce);
		this.video.addEventListener('playing', this.endStreamWhenCannotReconnectDebounce.cancel);
	};

	private resetEventEmitter() {
		this.eventEmitter.removeAllListeners();
		// "error" event type is treated as a special case and has to have at least one listener or it can crash the whole process
		// https://nodejs.org/api/events.html#events_error_events
		this.eventEmitter.addListener('error', () => {
			/* do nothing */
		});
	}

	private ifChromeLogWarningWithLinkToDocs() {
		const isChrome = !!(<any>this.window).chrome;
		if (isChrome) {
			console.warn(
				'It looks like an error occurred during video playback. ' +
					'View some of the common causes here https://docs.signageos.io/hc/en-us/articles/4405238997138.',
			);
		}
	}

	private prepareSequenceListeners() {
		return new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			let canPlayThroughListener: any, emptiedListener: any, preloadEnded: any;

			canPlayThroughListener = () => {
				this.video.removeEventListener('canplaythrough', canPlayThroughListener);
				this.video.removeEventListener('emptied', emptiedListener);
				this.video.removeEventListener('stalled', preloadEnded);
				resolve();
			};
			emptiedListener = () => {
				this.video.removeEventListener('emptied', emptiedListener);
				this.video.removeEventListener('canplaythrough', canPlayThroughListener);
				this.video.removeEventListener('stalled', preloadEnded);
				reject(new Error('Video status changed before it could finish prepare'));
			};

			const loadstartListener = () => {
				this.video.removeEventListener('loadstart', loadstartListener);
				this.video.addEventListener('canplaythrough', canPlayThroughListener);
				this.video.addEventListener('emptied', emptiedListener);
			};

			preloadEnded = () => {
				this.video.removeEventListener('loadstart', loadstartListener);
				this.video.removeEventListener('stalled', preloadEnded);
				this.video.removeEventListener('emptied', emptiedListener);
				this.video.removeEventListener('canplaythrough', canPlayThroughListener);
				reject(new Error("Couldn't connect to video"));
			};

			this.video.addEventListener('loadstart', loadstartListener);
			this.video.addEventListener('stalled', preloadEnded);
		});
	}

	private debugVideo() {
		const events = [
			'canplay',
			'canplaythrough',
			'durationchange',
			'emptied',
			'ended',
			'error',
			'loadeddata',
			'loadedmetadata',
			'loadstart',
			'pause',
			'play',
			'playing',
			'progress',
			'ratechange',
			'seeked',
			'seeking',
			'stalled',
			'suspend',
			'timeupdate',
			'volumechange',
			'waiting',
			'complete',
		];

		for (const event of events) {
			this.video.addEventListener(event, (...args: any[]) => {
				logDebug(`Video event: ${event}`, ...args);
			});
		}
	}
}
