export const isSignageOsSymbol = Symbol('sos');

declare global {
	interface WindowSosSnippet {
		init: (options?: SnippetOptions) => Promise<void>;
	}

	interface Window {
		sos: {
			snippet: WindowSosSnippet;
		};
	}
	// eslint-disable-next-line no-var
	var window: Window & typeof globalThis;
}

export interface SnippetOptions {
	autoVerification?: {
		organizationUid: string;
		deviceName?: string;
	};
}

export const extendWindowSos = <T extends keyof Window['sos']>(
	property: T,
	value: Window['sos'][T],
	windowObject: Partial<Window> = window,
) => {
	if (windowObject.sos && !(windowObject.sos as any)[isSignageOsSymbol]) {
		throw new Error('Unable to create window.sos object, different property sos already exists on window');
	}

	if (windowObject.sos === undefined) {
		windowObject.sos = {} as typeof window.sos;
		Object.defineProperty(windowObject.sos, isSignageOsSymbol, { value: true });
	}

	windowObject.sos[property] = value;
};

export const extendWindowSosSnippet = (snippet: Omit<WindowSosSnippet, 'defer'>, windowObject: Partial<Window> = window) => {
	let initialized = false;

	const init = snippet.init;

	extendWindowSos(
		'snippet',
		{
			...snippet,
			init: async (options?: SnippetOptions) => {
				if (initialized) {
					console.warn('initSnippet was called multiple times, ignoring the excess calls');
					return;
				}
				initialized = true;
				return init(options);
			},
		},
		windowObject,
	);

	windowObject?.addEventListener?.('DOMContentLoaded', () => {
		// wait for a moment so user defined DOMContentLoaded listeners run first
		setTimeout(() => {
			if ('__SOS_AUTO_VERIFICATION' in windowObject && !initialized) {
				console.warn(
					[
						'Using snippet without sos.snippet.init() is deprecated, __SOS_AUTO_VERIFICATION variable and use please run sos.snippet.init() before or on DOMContentLoaded.',
						'https://docs.signageos.io/hc/en-us/articles/4405556589458-Snippet-Provisioning',
					].join('\n'),
				);
				init();
			}
		}, 50);
	});
};
