import { takeEveryAndBindWhenPlatform } from '../Socket/socketActionCreator';
import { put } from 'redux-saga/effects';
import { StartTestSuite, TestSucceeded, TestFailed, TestSkipped, TestResult } from '@signageos/actions/dist/Device/Test/deviceTestActions';
import { TriggerAppletOff, TriggerAppletOn } from '../Front/Device/Power/devicePowerActionActions';
import { runTests } from './TestFramework';
import * as t from './TestFramework';

export function* runTestsCallback(testSuitePromises: Promise<t.Describe>[], action: StartTestSuite) {
	const uid = action.uid;
	const uniqueRequestedTests = action.tests.filter((elem: string, pos: number) => {
		return action.tests.indexOf(elem) === pos;
	});

	if (uniqueRequestedTests.length === 0) {
		return;
	}

	const processedTests: string[] = [];

	yield put<TriggerAppletOff>({ type: TriggerAppletOff });
	try {
		for (let testSuitePromise of testSuitePromises) {
			const testSuite: t.Describe = yield testSuitePromise;

			if (!testSuite || !testSuite.title) {
				throw new Error('Invalid test - no describe defined');
			}

			// ignore test that was not requested
			if (action.tests.indexOf(testSuite.title) < 0) {
				continue;
			}

			processedTests.push(testSuite.title);
			const result: TestResult = yield runTests(testSuite);

			// mark failed when at least one test failed
			if (result.failed > 0) {
				yield put<TestFailed>({ type: TestFailed, uid, identifier: result.title, result });
				continue;
			}

			// mark as skipped only when all tests are skipped
			if (result.skipped === result.total) {
				yield put<TestSkipped>({ type: TestSkipped, uid, identifier: result.title, result });
				continue;
			}

			yield put<TestSucceeded>({ type: TestSucceeded, uid, identifier: result.title, result });
		}
	} catch (error) {
		console.error('runTestsCallback: ', error);
	} finally {
		for (let unsupportedTest of uniqueRequestedTests) {
			if (processedTests.indexOf(unsupportedTest) < 0) {
				console.warn('unknown test requested:', unsupportedTest);
			}
		}

		yield put<TriggerAppletOn>({ type: TriggerAppletOn });
	}
}

export function* runTestsSaga(tests: Promise<t.Describe>[]) {
	yield takeEveryAndBindWhenPlatform(StartTestSuite, runTestsCallback, tests);
}
