import ISystemInfo, { ISystemMemory } from '../NativeDevice/ISystemInfo';
import { checksumString } from '../Hash/checksum';

export default class WebBrowserEmulatorSystemInfo implements ISystemInfo {
	constructor(private window: Window) {}

	public async getModel(): Promise<string> {
		return this.window.navigator.platform;
	}

	public async getSerialNumber(): Promise<string> {
		return checksumString(this.window.navigator.userAgent, 12).toUpperCase();
	}

	public async getCpuUsage(): Promise<number> {
		throw new Error('Not implemented');
	}

	public async getMemoryUsage(): Promise<ISystemMemory> {
		throw new Error('Not implemented');
	}

	public async getOSVersion(): Promise<string> {
		const navigator = this.window.navigator;

		let platform = navigator.platform;
		let platformVersion = tryGetPlatformVersionFromUserAgent(navigator.userAgent) ?? '0.0';

		if (isNavigatorUAData(navigator)) {
			const platformObject = await navigator.userAgentData.getHighEntropyValues(['platform', 'platformVersion']);
			platform = platformObject.platform;
			platformVersion = platformObject.platformVersion;
		}

		if (platform === 'Windows') {
			return getOSWindowsVersion(platformVersion);
		} else {
			return platformVersion;
		}
	}

	public async getBrand(): Promise<string> {
		return 'signageOS';
	}

	public async getOsName(): Promise<string> {
		// In the future we would like to return a Browser name and its version parsing
		// the user agent string using a library like: https://www.npmjs.com/package/ua-parser-js
		return 'WebBrowser';
	}
}

async function getOSWindowsVersion(platformVersion: string): Promise<string> {
	const majorPlatformVersion = parseInt(platformVersion.split('.')[0]);
	if (majorPlatformVersion >= 13) {
		return '11';
	} else if (majorPlatformVersion > 0) {
		return '10';
	} else {
		return '7,8,8.1';
	}
}

interface HighEntropyValues {
	platform: string;
	platformVersion: string;
}

interface NavigatorUAData extends Navigator {
	userAgentData: {
		getHighEntropyValues: (keys: (keyof HighEntropyValues)[]) => Promise<HighEntropyValues>;
	};
}

function isNavigatorUAData(navigator: Navigator): navigator is NavigatorUAData {
	return 'userAgentData' in navigator;
}

/**
 * Examples:
 * "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:109.0) Gecko/20100101 Firefox/115.0"  ->  "10.15"
 * "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/114.0.0.0 Safari/537.36"  ->  "10.15"
 * "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/114.0.0.0 Safari/537.36 Edg/114" ->  "10.0"
 * "Mozilla/5.0 (Linux; Android 13; SM-S908B) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/112.0.0.0 Mobile Safari/537.36" ->  "13"
 */
export function tryGetPlatformVersionFromUserAgent(userAgent: string) {
	let platformVersion = userAgent.match(/\([^)]*\s(\d+((\.|_)\d+)?)[^)]*\)/)?.[1]?.replace(/_+/g, '.');
	return platformVersion ?? null;
}
