import { EventEmitter } from 'events';
import { locked } from '@signageos/lib/es6/Lock/lockedDecorator';
import { debug } from '@signageos/lib/es6/Debug/debugDecorator';
import ISynchronizer, {
	BroadcastedValue,
	BroadcastedValueArgs,
	BroadcastedValueCallback,
	ClosedCallback,
	GroupLeftCallback,
	GroupStatus,
	GroupStatusCallback,
	JoinGroupArgs,
	SynchronizerEvent,
	SynchronizerEventCallback,
	WaitArgs,
} from '../ISynchronizer';
import IDriver from '../../NativeDevice/Front/IFrontDriver';
import Debug from 'debug';

const DEBUG_NAMESPACE = '@signageos/front-display:Front:Applet:Sync:SocketSynchronizer';
const logDebug = Debug(DEBUG_NAMESPACE);
const logDebugSocket = Debug(DEBUG_NAMESPACE + ':Socket');

interface IGroup {
	isOnline: boolean;
	deviceIdentification: string;
	lastStatus?: GroupStatus;
}

/**
 * API to synchronize content with other devices in the network.
 * When online
 *     - If everything is fine and device is always online, the logic is simple - initialize group on the server to determine,
 *       which devices belong together and whenever wait() is called, wait until all other devices are waiting too
 *       and resolve it on all simultaniously.
 * When offline
 *     - If the device goes offline in any point in the process, it silently switches to offline mode, which means that
 *       the SocketSynchronizer is itself acting like a mock server and simply validates and confirms all the requests to
 *       the API immediately to maintain smooth and seamless operation.
 *     - SocketSynchronizer also buffers all the initialized groups while offline and when it goes back online, it dumps the buffer
 *       to the server to synchronize the internal state.
 */
class SocketSynchronizer implements ISynchronizer {
	private socket: SocketIOClient.Socket | undefined;
	private connected: boolean = false;
	private groups: {
		[groupName: string]: IGroup;
	} = {};
	private emitter: EventEmitter = new EventEmitter();

	constructor(
		private getNativeDriver: () => IDriver,
		private createSocket: (serverUri: string) => SocketIOClient.Socket,
		private defaultServerUri: string,
	) {}

	@debug(DEBUG_NAMESPACE)
	public async connect(serverUri?: string): Promise<void> {
		if (this.socket) {
			await this.close();
		}
		this.initSocket(serverUri || this.defaultServerUri);
	}

	@debug(DEBUG_NAMESPACE)
	public async close(): Promise<void> {
		if (!this.socket) {
			throw new Error("Can't close when not connected");
		}
		this.socket.close();
		this.socket = undefined;
		this.connected = false;

		Object.keys(this.groups).forEach((groupName: string) => {
			this.emitter.emit(SynchronizerEvent.GroupLeft, groupName);
		});
		this.groups = {};
	}

	@locked('group', { scope: 'instance' })
	@debug(DEBUG_NAMESPACE)
	public async joinGroup({ groupName, deviceIdentification }: JoinGroupArgs): Promise<void> {
		if (!this.socket) {
			throw new Error('Not connected. Call connect() first.');
		}
		const deviceId = deviceIdentification !== undefined ? deviceIdentification : Math.random().toString(32);

		if (this.connected) {
			await this.joinGroupOnline(groupName, deviceId);
		} else {
			this.joinGroupOffline(groupName, deviceId);
		}
	}

	@locked('group', { scope: 'instance' })
	@debug(DEBUG_NAMESPACE)
	public async leaveGroup(groupName: string): Promise<void> {
		if (!this.socket) {
			throw new Error('Not connected. Call connect() first.');
		}
		if (!this.groups[groupName]) {
			throw new Error(`Group ${groupName} isn't joined`);
		}

		if (this.connected) {
			await this.leaveGroupOnline(groupName);
		} else {
			this.leaveGroupOffline(groupName);
		}
	}

	public async getDeviceIdentification(groupName: string) {
		if (this.groups[groupName]) {
			return this.groups[groupName].deviceIdentification;
		} else {
			return undefined;
		}
	}

	public async isConnected() {
		return !!this.socket;
	}

	@debug(DEBUG_NAMESPACE)
	public async wait({ groupName, data, timeoutMs }: WaitArgs): Promise<unknown> {
		if (!this.socket) {
			throw new Error('Not connected. Call connect() first.');
		}
		if (typeof timeoutMs !== 'undefined' && timeoutMs <= 0) {
			throw new Error('Timeout must be greater than 0');
		}
		if (this.connected) {
			return await this.callOnlineWait(groupName, data, timeoutMs);
		} else {
			return this.callOfflineWait(groupName, data);
		}
	}

	@debug(DEBUG_NAMESPACE)
	public async cancelWait(_groupName: string): Promise<void> {
		throw new Error('Not implemented yet');
	}

	@debug(DEBUG_NAMESPACE)
	public async broadcastValue({ groupName, key, value }: BroadcastedValueArgs): Promise<void> {
		if (!this.socket) {
			throw new Error('Not connected. Call connect() first.');
		}
		if (!this.groups[groupName]) {
			throw new Error('Group ' + groupName + " wasn't initialized");
		}
		this.socket.emit('request_set_value', { groupName, key, value });
		// if not connected, emit the value back to oneself
		if (!this.connected) {
			this.emitValue(groupName, key, value);
		}
	}

	public async isMaster(groupName: string): Promise<boolean> {
		const group = this.groups[groupName];

		if (group && group.lastStatus) {
			return group.lastStatus.isMaster;
		} else {
			return true;
		}
	}

	public addListener(event: SynchronizerEvent.GroupStatus, listener: GroupStatusCallback): void;
	public addListener(event: SynchronizerEvent.GroupLeft, listener: GroupLeftCallback): void;
	public addListener(event: SynchronizerEvent.BroadcastedValue, listener: BroadcastedValueCallback): void;
	public addListener(event: SynchronizerEvent.Closed, listener: ClosedCallback): void;
	public addListener(event: SynchronizerEvent, listener: SynchronizerEventCallback): void {
		this.emitter.addListener(event, listener);
	}

	public removeListener(event: SynchronizerEvent.GroupStatus, listener: GroupStatusCallback): void;
	public removeListener(event: SynchronizerEvent.GroupLeft, listener: GroupLeftCallback): void;
	public removeListener(event: SynchronizerEvent.BroadcastedValue, listener: BroadcastedValueCallback): void;
	public removeListener(event: SynchronizerEvent.Closed, listener: ClosedCallback): void;
	public removeListener(event: SynchronizerEvent, listener: SynchronizerEventCallback): void {
		this.emitter.removeListener(event, listener);
	}

	private initSocket(serverUri: string) {
		this.socket = this.createSocket(serverUri);
		this.debugSocket(this.socket, serverUri);

		this.socket.on('connect', async () => {
			// flush all offline groups to the server
			for (let groupName of Object.keys(this.groups)) {
				if (this.groups[groupName].isOnline === false) {
					try {
						await this.joinGroupOnline(groupName, this.groups[groupName].deviceIdentification);
					} catch (error) {
						console.error('flush offline initialized group to the server', groupName, error);
					}
				}
			}

			this.connected = true;
		});
		this.socket.on('disconnect', () => {
			this.connected = false;
			// when device disconnects, all initialized groups are now offline and has to be flushed to the server on reconnection
			for (let groupName of Object.keys(this.groups)) {
				this.groups[groupName].isOnline = false;
				this.groups[groupName].lastStatus = undefined;
			}
		});
		this.socket.on('set_value', (groupName: string, key: string, value: unknown) => {
			if (this.groups[groupName]) {
				const broadcastedValue: BroadcastedValue = { groupName, key, value };
				this.emitter.emit(SynchronizerEvent.BroadcastedValue, broadcastedValue);
			}
		});
		this.socket.on('device_status', (connectedPeers: string[], groupName: string) => {
			if (this.groups[groupName]) {
				const master = this.pickMaster(connectedPeers);
				const isMaster = master === this.groups[groupName].deviceIdentification;
				const groupStatus: GroupStatus = { groupName, connectedPeers, isMaster };
				this.groups[groupName].lastStatus = groupStatus;
				this.emitter.emit(SynchronizerEvent.GroupStatus, groupStatus);
			}
		});
	}

	private debugSocket(socket: SocketIOClient.Socket, serverUri: string) {
		logDebugSocket('open socket', { serverUri });
		socket.on('connect', () => logDebugSocket('connect'));
		socket.on('connect_error', (error: Error) => logDebugSocket('connect_error', error));
		socket.on('connect_timeout', () => logDebugSocket('connect_timeout'));
		socket.on('error', (error: Error) => logDebugSocket('error', error));
		socket.on('disconnect', () => logDebugSocket('disconnect'));
		socket.on('reconnect_attempt', () => logDebugSocket('reconnect_attempt'));
		socket.on('reconnecting', () => logDebugSocket('reconnecting'));
		socket.on('reconnect_error', (error: Error) => logDebugSocket('reconnect_error', error));
		socket.on('reconnect_failed', () => logDebugSocket('reconnect_failed'));
		socket.on('ping', () => logDebugSocket('ping'));
		socket.on('pong', () => logDebugSocket('pong'));
	}

	private async joinGroupOnline(groupName: string, deviceIdentification: string) {
		const nativeDriver = this.getNativeDriver();
		const deviceUid = await nativeDriver.getDeviceUid();

		await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			this.socket!.emit('join_group', { deviceUid, groupName, deviceIdentification }, (error?: Error) => {
				if (error) {
					reject(error);
				} else {
					resolve();
				}
			});
		});

		this.groups[groupName] = {
			isOnline: true,
			deviceIdentification,
		};
	}

	private joinGroupOffline(groupName: string, deviceIdentification: string) {
		this.groups[groupName] = {
			isOnline: false,
			deviceIdentification,
		};
	}

	private async leaveGroupOnline(groupName: string) {
		await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			this.socket!.emit('leave_group', { groupName }, (error?: Error) => {
				if (error) {
					reject(error);
				} else {
					resolve();
				}
			});
		});

		delete this.groups[groupName];
	}

	private leaveGroupOffline(groupName: string) {
		delete this.groups[groupName];
	}

	private async callOnlineWait(groupName: string, data?: unknown, timeout?: number) {
		return await new Promise((resolve: (data?: unknown) => void, reject: (error: Error) => void) => {
			let syncConfirmedListener: Function;
			let disconnectListener: Function;
			let timeoutHandler: ReturnType<typeof setTimeout> | undefined;

			syncConfirmedListener = (confirmedGroupName: string, confirmData?: unknown) => {
				if (confirmedGroupName === groupName) {
					if (typeof timeoutHandler !== 'undefined') {
						clearTimeout(timeoutHandler);
					}
					this.socket!.off('disconnect', disconnectListener);
					this.socket!.off('sync_confirmed', syncConfirmedListener);
					resolve(confirmData);
				}
			};
			disconnectListener = () => {
				if (typeof timeoutHandler !== 'undefined') {
					clearTimeout(timeoutHandler);
				}
				this.socket!.off('disconnect', disconnectListener);
				this.socket!.off('sync_confirmed', syncConfirmedListener);
				// if it goes offline, it should automatically confirm all waits as if nothing happened
				try {
					const confirmData = this.callOfflineWait(groupName, data);
					resolve(confirmData);
				} catch (error) {
					reject(error);
				}
			};

			this.socket!.on('sync_confirmed', syncConfirmedListener);
			this.socket!.on('disconnect', disconnectListener);

			this.socket!.emit('sync_request', { groupName, data }, (error?: Error) => {
				if (error) {
					this.socket!.off('disconnect', disconnectListener);
					this.socket!.off('sync_confirmed', syncConfirmedListener);
					reject(error);
				}
			});

			if (typeof timeout !== 'undefined') {
				timeoutHandler = setTimeout(() => {
					syncConfirmedListener(groupName, data);
					this.socket!.emit('sync_request_cancel', { groupName });
				}, timeout);
			}
		});
	}

	private callOfflineWait(groupName: string, data?: unknown) {
		if (typeof this.groups[groupName] === 'undefined') {
			throw new Error('Group ' + groupName + "wasn't initialized");
		} else {
			logDebug('offline wait confirmed for group ' + groupName);
			return data;
		}
	}

	private emitValue(groupName: string, key: string, value: unknown) {
		if (!this.groups[groupName]) {
			throw new Error('Group ' + groupName + "wasn't initialized");
		}

		this.emitter.emit(SynchronizerEvent.BroadcastedValue, { groupName, key, value });
	}

	/**
	 * Picks a master from a list of peers in a deterministic way
	 */
	private pickMaster(peers: string[]) {
		return [...peers].sort()[0];
	}
}

export default SocketSynchronizer;
