import { EventEmitter } from 'events';
import Debug from 'debug';
import { debug } from '@signageos/lib/dist/Debug/debugDecorator';
import { locked } from '@signageos/lib/es6/Lock/lockedDecorator';
import { IPolymorphicSynchronizer, SynchronizerType } from './IPolymorphicSynchronizer';
import ISynchronizer, {
	SynchronizerEvent,
	JoinGroupArgs,
	WaitArgs,
	BroadcastedValueArgs,
	GroupStatus,
	BroadcastedValue,
	SynchronizerEventCallback,
} from '../ISynchronizer';

const DEBUG_NAMESPACE = '@signageos/front-display:Synchronization:PolymorphicSynchronizer';
const logDebug = Debug(DEBUG_NAMESPACE);

/**
 * Allows to select from multiple synchronizer implementations at runtime.
 *
 * The particular implementation is selected by the connect() method.
 * Once it's connected, all the methods will adhere to that implementation.
 * Only one of the implementations should be used at a time.
 * Call close() to stop using the current implementation.
 * After calling close(), the connect() method can be used to select a different implementation.
 */
export class PolymorphicSynchronizer implements IPolymorphicSynchronizer {
	private emitter: EventEmitter = new EventEmitter();
	private currentType: SynchronizerType | null = null;

	/**
	 * @param synchronizers A map of synchronizer implementations. The key is the type of the synchronizer.
	 *                      The first synchronizer in the map will be used as the default synchronizer.
	 */
	constructor(private synchronizers: Map<SynchronizerType, ISynchronizer>) {
		if (this.synchronizers.size === 0) {
			throw new Error('No synchronizers provided');
		}

		this.onGroupStatus = this.onGroupStatus.bind(this);
		this.onGroupLeft = this.onGroupLeft.bind(this);
		this.onBroadcastedValue = this.onBroadcastedValue.bind(this);
		this.onClosed = this.onClosed.bind(this);
	}

	/**
	 * Selects the synchronizer implementation to use and makes it connect to the sync network
	 * @param serverUri The URI of the sync server to connect to. If not specified, the default sync server will be used.
	 *                  Only applicable to the socket synchronizer.
	 * @param type The type of the synchronizer to use. If not specified, the default synchronizer will be used.
	 */
	@debug(DEBUG_NAMESPACE)
	@locked('sync', { scope: 'instance' })
	public async connect(serverUri?: string | undefined, type?: SynchronizerType | undefined): Promise<void> {
		if (this.currentType) {
			throw new Error(`Already using ${type} synchronizer`);
		}

		type = type ?? this.getDefaultSynchronizerType();

		const synchronizer = this.getSynchronizerByType(type);
		await synchronizer.connect(serverUri);
		this.listenToSynchronizerEvents(synchronizer);

		this.currentType = type;
	}

	@debug(DEBUG_NAMESPACE)
	@locked('sync', { scope: 'instance' })
	public async close(): Promise<void> {
		const promises: Promise<void>[] = [];

		for (const synchronizer of this.synchronizers.values()) {
			promises.push(this.closeSynchronizerIfConnected(synchronizer));
		}

		await Promise.all(promises);
		this.currentType = null;
	}

	@debug(DEBUG_NAMESPACE)
	public async isConnected(): Promise<boolean> {
		if (!this.currentType) {
			return false;
		}

		const synchronizer = this.getSynchronizerByType(this.currentType);
		return synchronizer.isConnected();
	}

	@debug(DEBUG_NAMESPACE)
	@locked('sync', { scope: 'instance' })
	public async joinGroup(args: JoinGroupArgs): Promise<void> {
		const synchronizer = this.getCurrentSynchronizer();
		await synchronizer.joinGroup(args);
	}

	@debug(DEBUG_NAMESPACE)
	@locked('sync', { scope: 'instance' })
	public async leaveGroup(groupName: string): Promise<void> {
		const synchronizer = this.getCurrentSynchronizer();
		await synchronizer.leaveGroup(groupName);
	}

	public async getDeviceIdentification(groupName: string) {
		const synchronizer = this.getCurrentSynchronizer();
		return await synchronizer.getDeviceIdentification(groupName);
	}

	@debug(DEBUG_NAMESPACE)
	public async wait(args: WaitArgs): Promise<unknown> {
		const synchronizer = this.getCurrentSynchronizer();
		return synchronizer.wait(args);
	}

	@debug(DEBUG_NAMESPACE)
	public async cancelWait(groupName: string): Promise<void> {
		const synchronizer = this.getCurrentSynchronizer();
		await synchronizer.cancelWait(groupName);
	}

	@debug(DEBUG_NAMESPACE)
	public async broadcastValue(args: BroadcastedValueArgs): Promise<void> {
		const synchronizer = this.getCurrentSynchronizer();
		await synchronizer.broadcastValue(args);
	}

	@debug(DEBUG_NAMESPACE)
	public async isMaster(groupName: string): Promise<boolean> {
		const synchronizer = this.getCurrentSynchronizer();
		return await synchronizer.isMaster(groupName);
	}

	public addListener(event: SynchronizerEvent, listener: SynchronizerEventCallback): void {
		this.emitter.addListener(event, listener);
	}

	public removeListener(event: SynchronizerEvent, listener: SynchronizerEventCallback): void {
		this.emitter.removeListener(event, listener);
	}

	private listenToSynchronizerEvents(synchronizer: ISynchronizer) {
		synchronizer.addListener(SynchronizerEvent.GroupStatus, this.onGroupStatus);
		synchronizer.addListener(SynchronizerEvent.GroupLeft, this.onGroupLeft);
		synchronizer.addListener(SynchronizerEvent.BroadcastedValue, this.onBroadcastedValue);
		synchronizer.addListener(SynchronizerEvent.Closed, this.onClosed);
	}

	private stopListeningToSynchronizerEvents(synchronizer: ISynchronizer) {
		synchronizer.removeListener(SynchronizerEvent.GroupStatus, this.onGroupStatus);
		synchronizer.removeListener(SynchronizerEvent.GroupLeft, this.onGroupLeft);
		synchronizer.removeListener(SynchronizerEvent.BroadcastedValue, this.onBroadcastedValue);
		synchronizer.removeListener(SynchronizerEvent.Closed, this.onClosed);
	}

	private onGroupStatus(deviceStatus: GroupStatus) {
		logDebug('got group status event', deviceStatus);
		this.emitter.emit(SynchronizerEvent.GroupStatus, deviceStatus);
	}

	private onGroupLeft(groupName: string) {
		logDebug('got group left event', groupName);
		this.emitter.emit(SynchronizerEvent.GroupLeft, groupName);
	}

	private onBroadcastedValue(broadcastedValue: BroadcastedValue) {
		logDebug('got broadcasted value event', broadcastedValue);
		this.emitter.emit(SynchronizerEvent.BroadcastedValue, broadcastedValue);
	}

	private async onClosed(error?: Error) {
		logDebug('got closed event', error);
		this.emitter.emit(SynchronizerEvent.Closed, error);

		try {
			await this.close();
		} catch (e) {
			console.error('Failed to close synchronizer', e);
		}
	}

	private getDefaultSynchronizerType(): SynchronizerType {
		return this.synchronizers.keys().next().value;
	}

	private getSynchronizerByType(type: SynchronizerType) {
		const synchronizer = this.synchronizers.get(type);
		if (!synchronizer) {
			throw new Error(`Unknown synchronizer type: ${type}`);
		}

		return synchronizer;
	}

	private getCurrentSynchronizer() {
		if (!this.currentType) {
			throw new Error('Not connected');
		}

		return this.getSynchronizerByType(this.currentType);
	}

	private async closeSynchronizerIfConnected(synchronizer: ISynchronizer) {
		try {
			if (await synchronizer.isConnected()) {
				await synchronizer.close();
			}
		} finally {
			this.stopListeningToSynchronizerEvents(synchronizer);
		}
	}
}
