import { EventEmitter } from 'events';
import Debug from 'debug';
import { locked } from '@signageos/lib/es6/Lock/lockedDecorator';
import { debug } from '@signageos/lib/dist/Debug/debugDecorator';
import { generateUniqueHash } from '@signageos/lib/dist/Hash/generator';
import wait from '@signageos/lib/dist/Timer/wait';
import { CancelablePromise } from '@signageos/lib/dist/Promise/cancelable';
import ISynchronizer, {
	BroadcastedValue,
	BroadcastedValueArgs,
	BroadcastedValueCallback,
	ClosedCallback,
	GroupLeftCallback,
	GroupStatus,
	GroupStatusCallback,
	JoinGroupArgs,
	SynchronizerEvent,
	SynchronizerEventCallback,
	WaitArgs,
} from '../ISynchronizer';
import { GroupEvent, GroupMember, IGroup } from '../Group/IGroup';
import { Group } from '../Group/Group';
import { WaitService } from '../Group/WaitService';
import { IWaitService } from '../Group/IWaitService';
import { BroadcastEvent, IBroadcastService } from '../Group/IBroadcastService';
import { BroadcastService } from '../Group/BroadcastService';
import { IPeerNetwork, PeerNetworkEvent } from '../../PeerNetwork/IPeerNetwork';

const DEBUG_NAMESPACE = '@signageos/front-display:Synchronization:P2PSynchronizer';
const logDebug = Debug(DEBUG_NAMESPACE);

interface JoinedGroup {
	group: IGroup;
	waitService: IWaitService;
	broadcastService: IBroadcastService;
	leave: () => Promise<void>;
}

export interface GroupFactoryArgs {
	name: string;
	groupName: string;
	peerNetwork: IPeerNetwork;
}

export type GroupFactory = (args: GroupFactoryArgs) => IGroup;

function defaultGroupFactory(args: GroupFactoryArgs) {
	return new Group(args);
}

export interface P2PSynchronizerOptions {
	/** how often should it emit device_status event with current status */
	deviceStatusInterval: number;
}

const defaultOptions: P2PSynchronizerOptions = {
	deviceStatusInterval: 30e3,
};

/**
 * Synchronizes devices via provided IPeerDiscoveryService
 *
 * The original purpose of this is to implement synchronization of devices in
 * local network via UDP without a need for an external synchronization server.
 */
export class P2PSynchronizer implements ISynchronizer {
	private groups: { [groupName: string]: JoinedGroup } = {};
	private options: P2PSynchronizerOptions;
	private emitter: EventEmitter = new EventEmitter();

	constructor(
		private peerNetwork: IPeerNetwork,
		options: Partial<P2PSynchronizerOptions> = {},
		private groupFactory: GroupFactory = defaultGroupFactory,
	) {
		this.options = { ...defaultOptions, ...options };
		this.handlePeerNetworkClosed = this.handlePeerNetworkClosed.bind(this);
		this.listenToPeerNetworkEvents();
	}

	@debug(DEBUG_NAMESPACE)
	@locked('sync', { scope: 'instance' })
	public async connect(_serverUri?: string | undefined): Promise<void> {
		await this.peerNetwork.start();
	}

	@debug(DEBUG_NAMESPACE)
	@locked('sync', { scope: 'instance' })
	public async close(): Promise<void> {
		const groupKeys = Object.keys(this.groups);
		for (let i = 0; i < groupKeys.length; i++) {
			try {
				const group = this.groups[groupKeys[i]];
				await group.leave();
			} catch (error) {
				// errors can only be logged but not thrown because the close has to finish
				// if group leave fails, it's probably because the peer network is already closed
				// or something even worse happened that we can't fix
				console.warn('Failed to leave group', error);
			}
		}

		if (this.peerNetwork.isStarted()) {
			await this.peerNetwork.stop();
		}

		this.groups = {};
	}

	public async isConnected(): Promise<boolean> {
		return this.peerNetwork.isStarted();
	}

	@debug(DEBUG_NAMESPACE)
	@locked('sync', { scope: 'instance' })
	public async joinGroup({ groupName, deviceIdentification }: JoinGroupArgs): Promise<void> {
		if (this.groups[groupName]) {
			console.warn(`Trying to join group ${groupName}, but it's already joined`);
			return;
		}

		const deviceId = deviceIdentification ?? generateUniqueHash(10);

		this.groups[groupName] = await this.createGroup({ deviceId, groupName });
	}

	@debug(DEBUG_NAMESPACE)
	@locked('sync', { scope: 'instance' })
	public async leaveGroup(groupName: string): Promise<void> {
		if (!this.groups[groupName]) {
			console.warn(`Trying to leave group ${groupName} but it's not joined`);
			return;
		}

		await this.groups[groupName].leave();
		delete this.groups[groupName];
	}

	public async getDeviceIdentification(groupName: string) {
		if (this.groups[groupName]) {
			return this.groups[groupName].group.getMe().name;
		} else {
			return undefined;
		}
	}

	@debug(DEBUG_NAMESPACE)
	public async wait({ groupName, data, timeoutMs }: WaitArgs) {
		if (!this.groups[groupName]) {
			throw new Error(`Group ${groupName} isn't initialized, call joinGroup() first`);
		}

		if (timeoutMs !== undefined && timeoutMs <= 0) {
			throw new Error('timeout must be greater than 0');
		}

		const waitService = this.groups[groupName].waitService;
		return await this.waitOrTimeout(waitService, data, timeoutMs);
	}

	@debug(DEBUG_NAMESPACE)
	public async cancelWait(groupName: string): Promise<void> {
		if (!this.groups[groupName]) {
			console.warn(`cancelWait can't be performed. Group ${groupName} isn't initialized, call joinGroup() first`);
			return;
		}

		const waitService = this.groups[groupName].waitService;
		await waitService.cancelWait();
	}

	@debug(DEBUG_NAMESPACE)
	public async broadcastValue({ groupName, key, value }: BroadcastedValueArgs): Promise<void> {
		if (!this.groups[groupName]) {
			throw new Error(`Group ${groupName} isn't initialized, call joinGroup() first`);
		}

		const broadcastService = this.groups[groupName].broadcastService;
		await broadcastService.broadcastValue(key, value);
	}

	@debug(DEBUG_NAMESPACE)
	public async isMaster(groupName: string): Promise<boolean> {
		const group = this.groups[groupName];
		if (!group) {
			return true;
		}

		const master = group.group.getMaster();
		const me = group.group.getMe();
		return master.id === me.id;
	}

	public addListener(event: SynchronizerEvent.GroupStatus, listener: GroupStatusCallback): void;
	public addListener(event: SynchronizerEvent.GroupLeft, listener: GroupLeftCallback): void;
	public addListener(event: SynchronizerEvent.BroadcastedValue, listener: BroadcastedValueCallback): void;
	public addListener(event: SynchronizerEvent.Closed, listener: ClosedCallback): void;
	public addListener(event: SynchronizerEvent, listener: SynchronizerEventCallback): void {
		this.emitter.addListener(event, listener);
	}

	public removeListener(event: SynchronizerEvent.GroupStatus, listener: GroupStatusCallback): void;
	public removeListener(event: SynchronizerEvent.GroupLeft, listener: GroupLeftCallback): void;
	public removeListener(event: SynchronizerEvent.BroadcastedValue, listener: BroadcastedValueCallback): void;
	public removeListener(event: SynchronizerEvent.Closed, listener: ClosedCallback): void;
	public removeListener(event: SynchronizerEvent, listener: SynchronizerEventCallback): void {
		this.emitter.removeListener(event, listener);
	}

	private async createGroup({ deviceId, groupName }: { deviceId: string; groupName: string }): Promise<JoinedGroup> {
		const group = this.groupFactory({
			name: deviceId,
			groupName,
			peerNetwork: this.peerNetwork,
		});

		await group.join();

		const waitService = new WaitService(group);
		await waitService.start();

		const broadcastService = new BroadcastService(group);
		broadcastService.start();

		const updateStatus = async () => {
			const members = group.getAllMembers();
			const connectedPeers = members.map((member: GroupMember) => member.name);
			const master = group.getMaster();
			const me = group.getMe();
			const isMaster = master.id === me.id;

			const groupStatus = { groupName, connectedPeers, isMaster };
			logDebug('group status changed', groupStatus);
			this.emitter.emit(SynchronizerEvent.GroupStatus, groupStatus as GroupStatus);
		};

		group.addListener(GroupEvent.MemberJoined, updateStatus);
		group.addListener(GroupEvent.MemberLeft, updateStatus);

		broadcastService.addListener(BroadcastEvent.Value, (key: string, value: unknown) => {
			const broadcastedValue: BroadcastedValue = { groupName, key, value };
			logDebug('received broadcasted value', broadcastedValue);
			this.emitter.emit(SynchronizerEvent.BroadcastedValue, broadcastedValue);
		});

		const interval = setInterval(updateStatus, this.options.deviceStatusInterval);

		const leave = async () => {
			group.removeListener(GroupEvent.MemberJoined, updateStatus);
			group.removeListener(GroupEvent.MemberLeft, updateStatus);
			clearInterval(interval);
			waitService.stop();
			broadcastService.stop();
			await group.leave();
			group.removeAllListeners();
		};

		return {
			group,
			waitService,
			broadcastService,
			leave,
		};
	}

	private async waitOrTimeout(waitService: IWaitService, data?: unknown, timeoutMs?: number) {
		const waitPromise = waitService.sendWaitMessageAndWaitForOthers(data);
		if (timeoutMs) {
			const timeoutPromise = this.resolveWithDataAfterTimeout(data, timeoutMs);
			try {
				return await Promise.race([waitPromise, timeoutPromise]);
			} finally {
				timeoutPromise.cancel();
			}
		} else {
			return await waitPromise;
		}
	}

	private resolveWithDataAfterTimeout(data: unknown, timeoutMs: number): CancelablePromise<unknown> {
		const waitPromise = wait(timeoutMs);
		const finalPromise = waitPromise.then(() => data);
		(finalPromise as any).cancel = () => waitPromise.cancel();
		return finalPromise as CancelablePromise<unknown>;
	}

	private listenToPeerNetworkEvents() {
		this.peerNetwork.addListener(PeerNetworkEvent.Closed, this.handlePeerNetworkClosed);
	}

	private async handlePeerNetworkClosed(error?: Error) {
		try {
			await this.close();
		} catch (e) {
			console.warn('Failed to close synchronizer', e);
		}

		this.emitter.emit(SynchronizerEvent.Closed, error);
	}
}
