import { EventEmitter } from 'events';
import Debug from 'debug';
import { locked } from '@signageos/lib/es6/Lock/lockedDecorator';
import { debug } from '@signageos/lib/es6/Debug/debugDecorator';
import { IWaitService } from './IWaitService';
import { Data, GroupEvent, GroupMember, IGroup } from './IGroup';

const DEBUG_NAMESPACE = '@signageos/front-display:Synchronization:WaitService';
const logDebug = Debug(DEBUG_NAMESPACE);

enum MessageType {
	Query = 'query',
	Wait = 'wait',
	NotWaiting = 'not_waiting',
}

interface QueryMessage {
	type: MessageType.Query;
}

interface WaitMessage {
	type: MessageType.Wait;
	data: unknown;
}

interface NotWaitingMessage {
	type: MessageType.NotWaiting;
}

function isMessage(message: unknown): message is { type: string } {
	return typeof message === 'object' && message !== null && 'type' in message;
}

function isQueryMessage(message: unknown): message is QueryMessage {
	return isMessage(message) && message.type === MessageType.Query;
}

function isWaitMessage(message: unknown): message is WaitMessage {
	return isMessage(message) && message.type === MessageType.Wait;
}

function isNotWaitingMessage(message: unknown): message is NotWaitingMessage {
	return isMessage(message) && message.type === MessageType.NotWaiting;
}

enum Event {
	AllPeersWaiting = 'all_peers_waiting',
	WaitCanceled = 'wait_canceled',
}

/**
 * WaitService handles synchronization of runtime with others in the network via wait interface
 */
export class WaitService implements IWaitService {
	private started: boolean = false;

	private currentWait: { waiting: boolean; data?: unknown } = { waiting: false };
	private waitingPeers: { [id: string]: unknown } = {};

	private emitter: EventEmitter = new EventEmitter();

	constructor(private group: IGroup) {
		this.handleData = this.handleData.bind(this);
		this.handlePeersChanged = this.handlePeersChanged.bind(this);
	}

	@debug(DEBUG_NAMESPACE)
	public async sendWaitMessageAndWaitForOthers(data: unknown) {
		this.logDebug('sendWaitMessageAndWaitForOthers', data);
		this.currentWait = { waiting: true, data };

		try {
			await this.sendWaitMessage(data);
		} catch (error) {
			this.currentWait = { waiting: false };
			throw error;
		}

		const me = this.group.getMe();
		this.waitingPeers[me.id] = data;

		try {
			return await this.waitForAllPeers();
		} finally {
			this.currentWait = { waiting: false };
		}
	}

	@debug(DEBUG_NAMESPACE)
	public async cancelWait(): Promise<void> {
		this.logDebug('cancelWait');
		this.emitter.emit(Event.WaitCanceled);
		await this.sendNotWaitingMessage();

		const me = this.group.getMe();
		delete this.waitingPeers[me.id];
	}

	@locked('start', { scope: 'instance' })
	@debug(DEBUG_NAMESPACE)
	public async start() {
		this.logDebug('start');
		if (this.started) {
			throw new Error('wait service already started');
		}

		await this.sendQueryMessage();
		this.group.addListener(GroupEvent.Data, this.handleData);
		this.group.addListener(GroupEvent.MemberLeft, this.handlePeersChanged);

		this.started = true;
	}

	@debug(DEBUG_NAMESPACE)
	public stop() {
		this.logDebug('stop');
		if (!this.started) {
			throw new Error("wait service isn't started");
		}

		this.group.removeListener(GroupEvent.Data, this.handleData);
		this.group.removeListener(GroupEvent.MemberLeft, this.handlePeersChanged);

		this.started = false;
	}

	private async handleData(data: Data) {
		const { from, data: message } = data;

		if (isQueryMessage(message)) {
			await this.handleQueryMessage(from, message);
		} else if (isWaitMessage(message)) {
			this.handleWaitMessage(from, message);
		} else if (isNotWaitingMessage(message)) {
			this.handleNotWaitingMessage(from, message);
		}
	}

	private async handleQueryMessage(from: GroupMember, message: QueryMessage) {
		this.logDebug('got query message from ' + from.id, JSON.stringify(message));
		if (this.currentWait.waiting) {
			await this.sendWaitMessage(this.currentWait.data);
		}
	}

	private async handleWaitMessage(from: GroupMember, message: WaitMessage) {
		this.logDebug('got wait message from ' + from.id, JSON.stringify(message));
		this.waitingPeers[from.id] = message.data;
		await this.emitEventIfAllPeersWaiting();
	}

	private handleNotWaitingMessage(from: GroupMember, message: NotWaitingMessage) {
		this.logDebug('got not waiting message from ' + from.id, JSON.stringify(message));
		delete this.waitingPeers[from.id];
		this.emitEventIfAllPeersWaiting();
	}

	private async handlePeersChanged() {
		this.logDebug('peers changed');
		await this.clearDeadWaitingPeers();
		await this.emitEventIfAllPeersWaiting();
	}

	private async emitEventIfAllPeersWaiting() {
		if (await this.allPeersWaiting()) {
			this.emitter.emit(Event.AllPeersWaiting);
		}
	}

	private async clearDeadWaitingPeers() {
		const me = this.group.getMe();
		const peers = this.group.getPeers();
		const groupPeerIds = peers.map((peer: GroupMember) => peer.id);

		// remove dead peers from the list of waiting peers
		for (const id of Object.keys(this.waitingPeers)) {
			if (id !== me.id && !groupPeerIds.includes(id)) {
				this.logDebug(`peer ${id} dead, removed from waiting peers`);
				delete this.waitingPeers[id];
			}
		}
	}

	private async sendQueryMessage() {
		const message: QueryMessage = {
			type: MessageType.Query,
		};

		this.logDebug('send query message', JSON.stringify(message));
		await this.group.sendGroupDataMessage(message);
	}

	private async sendWaitMessage(data: unknown) {
		const message: WaitMessage = {
			type: MessageType.Wait,
			data,
		};

		this.logDebug('send wait message', JSON.stringify(message));
		await this.group.sendGroupDataMessage(message);
	}

	private async sendNotWaitingMessage() {
		const message: NotWaitingMessage = {
			type: MessageType.NotWaiting,
		};

		await this.group.sendGroupDataMessage(message);
	}

	private waitForAllPeers() {
		return new Promise(async (resolve: (data: unknown) => void, reject: (error: Error) => void) => {
			let resolveWithMasterDataAndReset: () => Promise<void>;
			let rejectWithCanceledError: () => Promise<void>;

			const cleanupEventListeners = () => {
				this.emitter.removeListener(Event.AllPeersWaiting, resolveWithMasterDataAndReset);
				this.emitter.removeListener(Event.WaitCanceled, rejectWithCanceledError);
			};

			resolveWithMasterDataAndReset = async () => {
				cleanupEventListeners();
				this.logDebug('all peers waiting, resolve');
				const data = await this.getDataOfWaitingMaster();
				this.reset();
				resolve(data);
			};

			rejectWithCanceledError = async () => {
				cleanupEventListeners();
				this.logDebug('wait will be rejected because it was canceled');
				reject(new Error('Wait canceled'));
			};

			if (await this.allPeersWaiting()) {
				await resolveWithMasterDataAndReset();
				return;
			}

			const peers = await this.group.getPeers();
			const peersLeftToWaitCount = await this.countPeersLeftToWait();
			this.logDebug(`waiting for ${peersLeftToWaitCount} peers, total ${peers.length} peers in group`);

			this.emitter.once(Event.AllPeersWaiting, resolveWithMasterDataAndReset);
			this.emitter.once(Event.WaitCanceled, rejectWithCanceledError);
		});
	}

	private async countPeersLeftToWait() {
		const peers = await this.group.getPeers();
		const peersCount = peers.length + 1; // +1 to count me as well
		const waitingPeersCount = Object.keys(this.waitingPeers).length;
		this.logDebug(`countPeersLeftToWait peers=${peers.length}, total=${peersCount}, waiting=${waitingPeersCount}`);
		return peersCount - waitingPeersCount;
	}

	private async allPeersWaiting(): Promise<boolean> {
		const peersLeftToWaitCount = await this.countPeersLeftToWait();
		return peersLeftToWaitCount <= 0;
	}

	private async getDataOfWaitingMaster() {
		const master = await this.group.getMaster();
		return this.waitingPeers[master.id];
	}

	private reset() {
		this.waitingPeers = {};
	}

	private logDebug(...args: any[]) {
		logDebug(this.group.getGroupName(), ...args);
	}
}
