import { locked } from '@signageos/lib/es6/Lock/lockedDecorator';
import Debug from 'debug';
import { EventEmitter } from 'events';
import { unionBy } from 'lodash';
import { IPeerNetwork, PeerNetworkEvent, PeerNetworkMessage } from '../../PeerNetwork/IPeerNetwork';
import { DataCallback, Data as GroupData, GroupEvent, GroupMember, IGroup, MemberCallback } from './IGroup';

const debug = Debug('@signageos/front-display:Synchronization:Group');

enum MessageType {
	Query = 'query',
	JoinGroup = 'join_group',
	LeaveGroup = 'leave_group',
	// eslint-disable-next-line @typescript-eslint/no-shadow
	GroupData = 'group_data',
}

interface QueryMessage {
	type: MessageType.Query;
	groupName: string;
}

interface JoinGroupMessage {
	type: MessageType.JoinGroup;
	groupName: string;
	name: string;
	joinedAt: number;
}

interface LeaveGroupMessage {
	type: MessageType.LeaveGroup;
	groupName: string;
}

interface GroupDataMessage {
	type: MessageType.GroupData;
	groupName: string;
	data: unknown;
}

function isMessage(message: unknown): message is { type: string } {
	return typeof message === 'object' && message !== null && 'type' in message;
}

function isQueryMessage(message: unknown): message is QueryMessage {
	return isMessage(message) && message.type === MessageType.Query;
}

function isJoinGroupMessage(message: unknown): message is JoinGroupMessage {
	return isMessage(message) && message.type === MessageType.JoinGroup;
}

function isLeaveGroupMessage(message: unknown): message is LeaveGroupMessage {
	return isMessage(message) && message.type === MessageType.LeaveGroup;
}

function isGroupDataMessage(message: unknown): message is GroupDataMessage {
	return isMessage(message) && message.type === MessageType.GroupData;
}

/**
 * Group automatically handles membership in a group for synchronization purposes.
 */
export class Group implements IGroup {
	private memberName: string;
	private groupName: string;
	private peerNetwork: IPeerNetwork;
	private peers: GroupMember[] = [];

	private started: boolean = false;
	private joinedAt: Date = new Date();

	private emitter: EventEmitter = new EventEmitter();

	private sendQueryIntervalMs: number;
	private sendQueryInterval: ReturnType<typeof setInterval> | null = null;

	constructor({
		name,
		groupName,
		peerNetwork,
		sendQueryIntervalMs,
	}: {
		name: string;
		groupName: string;
		peerNetwork: IPeerNetwork;
		sendQueryIntervalMs?: number;
	}) {
		this.memberName = name;
		this.groupName = groupName;
		this.peerNetwork = peerNetwork;
		this.handleMessage = this.handleMessage.bind(this);
		this.handlePeersChanged = this.handlePeersChanged.bind(this);
		this.sendQueryIntervalMs = sendQueryIntervalMs || 30e3;
	}

	public getGroupName(): string {
		return this.groupName;
	}

	public getMe(): GroupMember {
		const me = this.peerNetwork.getMe();
		return {
			id: me.id,
			name: this.memberName,
			joinedAt: this.joinedAt,
		};
	}

	public getPeers(): GroupMember[] {
		return this.peers;
	}

	public getAllMembers(): GroupMember[] {
		const me = this.getMe();
		const peers = this.getPeers();
		return [me, ...peers];
	}

	public getMaster(): GroupMember {
		const allMembers = this.getAllMembers();
		// sort members by joinedAt
		const membersSorted = allMembers.sort((a, b) => a.joinedAt.getTime() - b.joinedAt.getTime());
		return membersSorted[0];
	}

	@locked('join', { scope: 'instance' })
	public async join() {
		if (this.started) {
			throw new Error('group already joined');
		}

		this.peerNetwork.addListener(PeerNetworkEvent.Message, this.handleMessage);
		this.peerNetwork.addListener(PeerNetworkEvent.PeersChanged, this.handlePeersChanged);

		this.joinedAt = new Date();

		try {
			await this.sendQueryMessage();
			await this.sendJoinGroupMessage();
		} catch (error) {
			// we need to start listening to the events before sending the messages to prevent race conditions
			// but if something fails, we need to cleanup
			this.peerNetwork.removeListener(PeerNetworkEvent.Message, this.handleMessage);
			this.peerNetwork.removeListener(PeerNetworkEvent.PeersChanged, this.handlePeersChanged);
			throw error;
		}

		this.startSendingQueryMessages();
		this.started = true;
	}

	public async leave() {
		if (!this.started) {
			throw new Error('group not joined');
		}

		this.stopSendingQueryMessages();

		this.peerNetwork.removeListener(PeerNetworkEvent.Message, this.handleMessage);
		this.peerNetwork.removeListener(PeerNetworkEvent.PeersChanged, this.handlePeersChanged);

		await this.sendLeaveGroupMessage();
		this.started = false;
	}

	public async sendGroupDataMessage(data: unknown) {
		if (!this.started) {
			throw new Error("can't send group message if group not joined");
		}

		const message: GroupDataMessage = {
			type: MessageType.GroupData,
			groupName: this.groupName,
			data,
		};

		this.logDebug('send group_data message', JSON.stringify(message));
		await this.peerNetwork.send(message);
	}

	public addListener(event: GroupEvent.MemberJoined, callback: MemberCallback): void;
	public addListener(event: GroupEvent.MemberLeft, callback: MemberCallback): void;
	public addListener(event: GroupEvent.Data, callback: DataCallback): void;
	public addListener(event: GroupEvent, callback: MemberCallback | DataCallback): void {
		this.emitter.addListener(event, callback);
	}

	public removeListener(event: GroupEvent.MemberJoined, callback: MemberCallback): void;
	public removeListener(event: GroupEvent.MemberLeft, callback: MemberCallback): void;
	public removeListener(event: GroupEvent.Data, callback: DataCallback): void;
	public removeListener(event: GroupEvent, callback: MemberCallback | DataCallback): void {
		this.emitter.removeListener(event, callback);
	}

	public removeAllListeners(event?: GroupEvent | undefined): void {
		if (event) {
			this.emitter.removeAllListeners(event);
		} else {
			this.emitter.removeAllListeners();
		}
	}

	private async handleMessage(data: PeerNetworkMessage) {
		const { fromId, message } = data;

		if (isQueryMessage(message)) {
			await this.handleQueryMessage(fromId, message);
		} else if (isJoinGroupMessage(message)) {
			this.handleJoinGroupMessage(fromId, message);
		} else if (isLeaveGroupMessage(message)) {
			this.handleLeaveGroupMessage(fromId, message);
		} else if (isGroupDataMessage(message)) {
			this.handleGroupDataMessage(fromId, message);
		}
	}

	private async handleQueryMessage(fromId: string, message: QueryMessage) {
		if (message.groupName === this.groupName) {
			this.logDebug('got query message from ' + fromId, JSON.stringify(message));
			await this.sendJoinGroupMessage();
		}
	}

	private handleJoinGroupMessage(fromId: string, message: JoinGroupMessage) {
		const { name, groupName, joinedAt } = message;

		if (groupName === this.groupName) {
			const peerAlreadyInList = this.findPeerById(fromId);
			if (!peerAlreadyInList) {
				this.logDebug('got join_group message from ' + fromId, JSON.stringify(message));

				const newMember = {
					id: fromId,
					name,
					joinedAt: new Date(joinedAt),
				};

				this.peers = unionBy(this.peers, [newMember], 'id'); // ensure unique list
				this.logDebug(`add new member ${fromId}, totalPeers=${this.peers.length}`);
				this.emitter.emit(GroupEvent.MemberJoined, newMember as GroupMember);
			}
		}
	}

	private handleLeaveGroupMessage(fromId: string, message: LeaveGroupMessage) {
		const { groupName } = message;

		if (groupName === this.groupName) {
			const groupPeer = this.findPeerById(fromId);

			if (groupPeer) {
				this.logDebug('got leave_group message from ' + fromId, JSON.stringify(message));
				this.peers = this.peers.filter((peer) => peer.id !== fromId);
				this.logDebug(`remove member ${fromId}, totalPeers=${this.peers.length}`);
				this.emitter.emit(GroupEvent.MemberLeft, groupPeer);
			} else {
				this.logDebug(`got leave_group message from ${fromId} but it's not known`, JSON.stringify(message));
			}
		}
	}

	private handleGroupDataMessage(fromId: string, message: GroupDataMessage) {
		if (message.groupName === this.groupName) {
			const groupPeer = this.findPeerById(fromId);

			if (groupPeer) {
				this.logDebug('got group_data message from ' + fromId, JSON.stringify(message));
				this.emitter.emit(GroupEvent.Data, {
					from: groupPeer,
					data: message.data,
				} as GroupData);
			} else {
				this.logDebug(`got group_data message from ${fromId} but it's not known`, JSON.stringify(message));
			}
		}
	}

	private async handlePeersChanged() {
		const peers = this.peerNetwork.getPeers();
		this.logDebug('Peers changed', peers);
		await this.clearDeadPeers();
	}

	private findPeerById(id: string) {
		return this.peers.find((peer: GroupMember) => peer.id === id);
	}

	private async clearDeadPeers() {
		const peers = this.peerNetwork.getPeers();
		const peersIds = peers.map((peer) => peer.id);

		// remove dead peers from the list of peers in the group
		const removedPeers = this.peers.filter((peer) => !peersIds.includes(peer.id));

		this.peers = this.peers.filter((peer) => peersIds.includes(peer.id));

		for (const removedPeer of removedPeers) {
			this.logDebug('peer removed', removedPeer.id);
			this.emitter.emit(GroupEvent.MemberLeft, removedPeer as GroupMember);
		}

		if (removedPeers.length > 0) {
			this.logDebug(`removed ${removedPeers.length} peers, totalPeers=${this.peers.length}`);
		}
	}

	/**
	 * Send query messages periodically to compensate for potential network issues
	 * If each peer only sends join_group and query once and someone is unable to receive it
	 * at that time because of network issues, they'll never send them again and
	 * there will be a partitioned group.
	 * Sending query messages periodically ensures that everyone will eventually
	 * receive the join_group message from everyone else, as long as the network issues get resolved.
	 */
	private startSendingQueryMessages() {
		this.sendQueryInterval = setInterval(() => this.sendQueryMessage(), this.sendQueryIntervalMs);
	}

	private stopSendingQueryMessages() {
		if (this.sendQueryInterval) {
			clearInterval(this.sendQueryInterval);
			this.sendQueryInterval = null;
		}
	}

	private async sendQueryMessage() {
		const message: QueryMessage = {
			type: MessageType.Query,
			groupName: this.groupName,
		};

		this.logDebug('send query message', JSON.stringify(message));
		await this.peerNetwork.send(message);
	}

	private async sendJoinGroupMessage() {
		const message: JoinGroupMessage = {
			type: MessageType.JoinGroup,
			groupName: this.groupName,
			name: this.memberName,
			joinedAt: this.joinedAt.getTime(),
		};

		this.logDebug('send join_group message', JSON.stringify(message));
		await this.peerNetwork.send(message);
	}

	private async sendLeaveGroupMessage() {
		const message: LeaveGroupMessage = {
			type: MessageType.LeaveGroup,
			groupName: this.groupName,
		};

		this.logDebug('send leave_group message', JSON.stringify(message));
		await this.peerNetwork.send(message);
	}

	private logDebug(...args: any[]) {
		debug(this.groupName, ...args);
	}
}
