import { bindAndTakeEvery } from './socketActionCreator';
import { call, take, put, fork, cancel, takeEvery, race } from 'redux-saga/effects';
import CircularJSON from 'circular-json';
import wait from '@signageos/lib/dist/Timer/wait';
import { createChannel, IChannel, takeEvery as takeEveryFromChannel } from '../ReduxSaga/channels';
import ISocket, { IErrorEvent, UnconfirmedMessageError, UndeliveredEmitError } from '@signageos/lib/dist/WebSocket/Client/ISocket';
import createSocketIOSocket from '@signageos/lib/dist/WebSocket/Client/SocketIO/createSocketIOSocket';
import { createSocket as createSocketWs } from '@signageos/lib/dist/WebSocket/Client/WS/createWSSocket';
import { createSocket as createSocketHttp } from './Http/createHttpSocket';
import IAction from '@signageos/actions/dist/IAction';
import IActionWithSource from './IActionWithSource';
import { PlatformConnect, SocketConnected, SocketDisconnected } from '@signageos/actions/dist/Socket/socketActions';
import IOfflineAction from '@signageos/actions/dist/IOfflineAction';
import { StopApplication } from '../Application/applicationActions';
import IBasicDriver from '../NativeDevice/IBasicDriver';
import { defineInvisibleProperty } from '../Object/property';
import IActionWithEmitter from '../Application/IActionWithEmitter';
import { now } from '@signageos/lib/dist/DateTime/dateTimeFactory';
import { EventEmitter } from 'events';
import { SocketConnected as SocketConnectedInternal, SocketConnectionFailed, BindAction, UnbindAction } from './socketActions';
import { checkServerTimeWhenRequested } from '../Device/DateTime/serverTimeCheckerSaga';
import Debug from 'debug';
import { createProgressiveWait } from '@signageos/lib/dist/Timer/progressiveWait';
import { resolveOnceEvent, rejectOnceEvent } from '../Util/promise';
import { shouldBeAddedToFastOfflineBuffer, shouldBeSentToServer } from './socketActionFilter';
import { SaveOfflineAction } from '../Offline/offlineActions';
import { SagaIterator, Task } from 'redux-saga';
import { isEmpty } from 'lodash';
import { PowerControl } from '../Management/Device/Power/PowerKernel';
import IConfig from '../Display/IConfig';
import { Disconnected } from '@signageos/actions/dist/Network/networkActions';

const debug = Debug('@signageos/front-display:Socket:socketSagas');

const MESSAGE_SERVER_AUTHENTICATE = 'server-authenticate';
const MESSAGE_SERVER_AUTHENTICATE_DONE = 'server-authenticate-done';
const MESSAGE_CLIENT_AUTHENTICATE = 'client-authenticate';
const MESSAGE_ACTION = 'action';
const messagesToServerQueue: [] = [];
let isHttpSocket: boolean = false;

export type SocketName = 'front' | 'management' | 'service' | 'unknown';

const FAST_OFFLINE_BUFFER_LIMIT = 100;
const ACTION_DELIVERY_TIMEOUT_MS = 30e3;
const HTTP_SOCKET_CHECK_INTERVAL_MS = 60e3;

type ISocketFactory = (
	baseUrl: string,
	onConnected: () => void,
	onDisconnected: () => void,
	onError: (error: IErrorEvent | UndeliveredEmitError | UnconfirmedMessageError) => void,
) => ISocket;

export function* socketPlatformCreate(
	getNativeDriver: () => IBasicDriver,
	socketName: SocketName,
	getDeviceTimestamp: () => Promise<number>,
	defaultConfig: IConfig,
	powerKernel?: PowerControl,
) {
	const actionChannel = createChannel<IActionWithSource & IActionWithEmitter>();
	const offlineActionChannel = createChannel<IActionWithSource & IOfflineAction>();
	const fastOfflineBuffer: Array<IActionWithSource & IActionWithEmitter> = [];
	const connectionEmitter = new EventEmitter();
	const progressiveWait = createProgressiveWait(5e3, 1.5, 5 * 60e3);
	let socketDriver: string;
	let platformUri: string;
	let checkTimeBeforeConnection: boolean;
	let connected: boolean = false;
	let exited: boolean = false;
	powerKernel?.doBeforeShutDown(waitUntilTheQueIsEmpty);

	function handleOfflineAction(action: (IActionWithSource & IActionWithEmitter) | any) {
		action.__dispatchedAt = action.__dispatchedAt ? action.__dispatchedAt : now().toDate();
		action.__offline = true;
		if (fastOfflineBuffer.length < FAST_OFFLINE_BUFFER_LIMIT && shouldBeAddedToFastOfflineBuffer(action)) {
			fastOfflineBuffer.push(action);
		} else {
			offlineActionChannel.put(action);
		}
	}

	yield takeEveryFromChannel(offlineActionChannel, function* (action: IActionWithSource & IOfflineAction) {
		if (action.type !== SaveOfflineAction) {
			yield put({ type: SaveOfflineAction, action } as SaveOfflineAction<typeof action>);
		}
	});

	function streamAction(action: IActionWithSource & IActionWithEmitter, actionDeliveryTimeoutMs: number) {
		if (connected) {
			const timeoutHandler = setTimeout(() => streamAction(action, actionDeliveryTimeoutMs), actionDeliveryTimeoutMs);
			action.__emitter.once('delivered', () => clearTimeout(timeoutHandler));
			actionChannel.put(action as IActionWithSource & IActionWithEmitter);
		} else if (action.type !== SaveOfflineAction) {
			handleOfflineAction(action);
		}
	}

	const boundActions: { [actionName: string]: number } = {};
	yield takeEvery(BindAction, function* (bindAction: BindAction): IterableIterator<void> {
		const actionName = bindAction.actionName;
		if (typeof boundActions[actionName] === 'undefined') {
			boundActions[actionName] = 1;
		} else {
			boundActions[actionName]++;
		}
	});
	yield takeEvery(UnbindAction, function* (unbindAction: UnbindAction): IterableIterator<void> {
		const actionName = unbindAction.actionName;
		if (boundActions[actionName] === 1) {
			delete boundActions[actionName];
		} else if (typeof boundActions[actionName] !== 'undefined') {
			boundActions[actionName]--;
		} else {
			debug("unbinding action that wasn't bound: " + actionName);
		}
	});
	yield takeEvery('*', function* (action: IActionWithSource & IActionWithEmitter): IterableIterator<any> {
		const timeout = getActionDeliveryTimeout(socketDriver);
		streamAction(action, timeout);
	});

	yield bindAndTakeEvery(SocketConnected, function* (): SagaIterator {
		const timeout = getActionDeliveryTimeout(socketDriver);
		let action: (IActionWithSource & IActionWithEmitter) | undefined;
		while ((action = fastOfflineBuffer.shift())) {
			streamAction(action, timeout);
		}
	});
	yield bindAndTakeEvery(StopApplication, function* (): SagaIterator {
		exited = true;
		connectionEmitter.emit('terminate');
	});
	yield bindAndTakeEvery(PlatformConnect, function* (): SagaIterator {
		connectionEmitter.emit('terminate');
	});

	yield bindAndTakeEvery(PlatformConnect, function* (connectAction: PlatformConnect): IterableIterator<any> {
		socketDriver = connectAction.driver;
		platformUri = connectAction.platformUri;
		checkTimeBeforeConnection = connectAction.checkTimeBeforeConnection;
	});
	yield bindAndTakeEvery(Disconnected, function* (): SagaIterator {
		connectionEmitter.emit('terminate');
	});
	yield fork(function* () {
		if (!platformUri) {
			yield take(PlatformConnect);
		}
		yield wait(1e3); // Wait before first socket connect, because device is always open until server configuration say different
		do {
			try {
				yield race([
					call(checkServerTimeWhenRequested, getNativeDriver, defaultConfig, checkTimeBeforeConnection, getDeviceTimestamp),
					call(() => rejectOnceEvent(connectionEmitter, 'terminate')),
				]);
				const duid: string = yield getNativeDriver().getDeviceUid();
				const createSocket = getSocketFactory(socketDriver);
				const socket = createSocket(
					platformUri + '/v2/',
					() => connectionEmitter.emit('connected'),
					() => connectionEmitter.emit('disconnected'),
					(error: any) => connectionEmitter.emit('error', error),
				);
				const connectedPromise = resolveOnceEvent(connectionEmitter, 'connected');
				const disconnectedPromise = rejectOnceEvent(connectionEmitter, 'disconnected');
				const terminatePromise = rejectOnceEvent(connectionEmitter, 'terminate');
				const errorPromise = rejectOnceEvent(connectionEmitter, 'error');
				const socketChannel = createSocketChannel(socket, MESSAGE_ACTION);
				debug('connecting to the socket');
				try {
					yield Promise.race([
						connectedPromise,
						disconnectedPromise, // if unable to connect (reject)
						errorPromise, // for sure when erred (reject)
					]);
				} catch (error) {
					yield put<SocketConnectionFailed>({
						type: SocketConnectionFailed,
						errorMessage: error?.message,
					});
					throw error;
				}
				debug('socket connected');
				yield put({ type: SocketConnectedInternal } as SocketConnectedInternal);
				try {
					yield Promise.race([
						disconnectedPromise, // for sure when disconnected
						errorPromise, // for sure when erred (reject)
						terminatePromise,
						authenticateSocket(socket, duid, socketName, Object.keys(boundActions)),
					]);
					debug('socket authenticated');
					const bindEffect: Task = yield fork(socketBindActions, socketChannel);
					const emitEffect: Task = yield fork(socketEmitActions, actionChannel, socket, MESSAGE_ACTION);
					connected = true;
					yield put({ type: SocketConnected } as SocketConnected);
					progressiveWait.reset();
					try {
						yield Promise.race([
							terminatePromise,
							disconnectedPromise, // for sure when disconnected
							errorPromise, // for sure when erred (reject)
						]);
					} finally {
						connected = false;
						yield cancel(bindEffect);
						yield cancel(emitEffect);
					}
				} finally {
					debug('socket disconnected');
					yield put({ type: SocketDisconnected } as SocketDisconnected);
					socket.removeAllListeners();
					socket.close(); // TODO should be always disconnected so should not be necessary to explicitly close
				}
			} catch (error) {
				debug('socketPlatformCreate failed', error);
			} finally {
				connectionEmitter.removeAllListeners('error');
				connectionEmitter.removeAllListeners('connected');
				connectionEmitter.removeAllListeners('disconnected');
				connectionEmitter.removeAllListeners('terminate');
			}
			yield progressiveWait.wait();
		} while (!exited);
	});
}

async function authenticateSocket(socket: ISocket, duid: string, name: SocketName, actions: string[]) {
	await new Promise<void>((resolve: () => void) => {
		socket.once(MESSAGE_SERVER_AUTHENTICATE, () => resolve());
	});
	socket.emit(MESSAGE_CLIENT_AUTHENTICATE, { duid, name, actions });
	await new Promise<void>((resolve: () => void) => {
		socket.once(MESSAGE_SERVER_AUTHENTICATE_DONE, () => resolve());
	});
}

function getSocketFactory(socketDriver: string): ISocketFactory {
	switch (socketDriver) {
		case 'ws':
			isHttpSocket = false;
			return createSocketWs;
		case 'http':
			isHttpSocket = true;
			return (...args: Parameters<ISocketFactory>) =>
				createSocketHttp(...args, {
					messagesToServerQueue,
					checkInterval: HTTP_SOCKET_CHECK_INTERVAL_MS,
				});
		case 'socket.io':
			isHttpSocket = false;
			return createSocketIOSocket;
		default:
			throw new Error(`Unknown socket driver ${socketDriver}`);
	}
}

function getActionDeliveryTimeout(socketDriver: string): number {
	switch (socketDriver) {
		case 'http':
			return HTTP_SOCKET_CHECK_INTERVAL_MS + ACTION_DELIVERY_TIMEOUT_MS;
		default:
			return ACTION_DELIVERY_TIMEOUT_MS;
	}
}

function* socketEmitActions(actionChannel: IChannel<IActionWithSource & IActionWithEmitter>, socket: ISocket, messageName: string) {
	function* takeAction(action: IActionWithSource & IActionWithEmitter) {
		try {
			yield call(emitAction, socket, messageName, action);
		} catch (error) {
			debug('emitAction failed', error);
		}
	}
	do {
		const action: IActionWithSource & IActionWithEmitter = yield call(actionChannel.take);
		yield fork(takeAction, action);
	} while (true);
}

function* socketBindActions(socketChannel: IChannel<IActionWithSource>) {
	function* takeAction(action: IActionWithSource) {
		yield put(action);
	}
	do {
		const action: IActionWithSource = yield call(socketChannel.take);
		yield fork(takeAction, action);
	} while (true);
}

function createSocketChannel(socket: ISocket, messageName: string) {
	const socketChannel = createChannel<IActionWithSource>();
	socket.on(messageName, (action: IActionWithSource) => {
		defineInvisibleProperty(action, '__source', socket);
		socketChannel.put(action);
	});
	return socketChannel;
}

function emitAction(socket: ISocket, messageName: string, action: IActionWithSource & IActionWithEmitter): Promise<void> {
	if (!isFromServer(action, socket) && shouldBeSentToServer(action)) {
		return new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			try {
				const actionMessage = JSON.parse(CircularJSON.stringify(action)) as IAction<string>;
				socket.emit(messageName, actionMessage, () => {
					action.__emitter.emit('delivered');
					resolve();
				});
			} catch (error) {
				reject(error);
			}
		});
	} else {
		action.__emitter.emit('delivered');
		return Promise.resolve();
	}
}

function isFromServer(action: IActionWithSource & IActionWithEmitter, socket: ISocket) {
	return action.__source === socket;
}

async function waitUntilTheQueIsEmpty() {
	return new Promise<void>((resolve) => {
		if (isHttpSocket) {
			const interval = setInterval(() => {
				if (isEmpty(messagesToServerQueue)) {
					clearInterval(interval);
					resolve();
				}
			}, 1e3);
		} else {
			resolve();
		}
	});
}
