import { put, take, call, takeEvery, fork, HelperWorkerParameters } from 'redux-saga/effects';
import { ActionMatchingPattern, SagaIterator } from '@redux-saga/types';
import { BindAction, UnbindAction } from './socketActions';
import { ChangeSubscriptionToOpen, ChangeSubscriptionToPlatform } from '../Device/Configuration/deviceConfigurationActions';
import { withDependencies } from '../DI/dependencyInjection';
import { SubscriptionType } from '../Display/IConfig';
import { StopApplication } from '../Application/applicationActions';
import { SagaIteratorWithPromises } from '../ReduxSaga/extendedEffects';

export const bindAndTakeEvery = <A extends string | string[], Fn extends (...args: unknown[]) => SagaIteratorWithPromises>(
	actionNames: A,
	saga: Fn,
	...args: HelperWorkerParameters<ActionMatchingPattern<A>, Fn>
) =>
	fork(function* (): SagaIterator {
		yield bindActionsUntilStopped(actionNames);
		yield takeEvery(actionNames, saga, ...args);
	});

/**
 * This function process actions and send them FROM server only when device is platform.
 * When device is open actions aren't process and then aren't send FROM server.
 */
export const bindAndTakeEveryOnlyWhenPlatform = <A extends string | string[], Fn extends (...args: unknown[]) => SagaIteratorWithPromises>(
	actionNames: A,
	saga: Fn,
	...args: HelperWorkerParameters<ActionMatchingPattern<A>, Fn>
) =>
	fork(
		withDependencies(['subscriptionType'], function* ({ subscriptionType }: { subscriptionType: string }): SagaIterator {
			if (subscriptionType === SubscriptionType.platform) {
				yield bindAndTakeEvery(actionNames, saga, ...args);
			}
		}),
	);

/**
 * Actions are send FROM server only when device is platform.
 * @deprecated Use withDependencies instead and all will be bound correctly
 */
export const bindWhenPlatform = <P extends string | string[]>(actionNames: P) =>
	fork(function* () {
		while (true) {
			yield take(ChangeSubscriptionToPlatform);
			yield bindActions(actionNames);
			yield take(ChangeSubscriptionToOpen);
			yield unbindActions(actionNames);
		}
	});

/**
 * This function process actions when device is platform or open.
 * Actions are send FROM server only when device is platform.
 */
export const takeEveryAndBindWhenPlatform = <P extends string | string[], Fn extends (...args: unknown[]) => SagaIteratorWithPromises>(
	actionNames: P,
	saga: Fn,
	...args: HelperWorkerParameters<ActionMatchingPattern<P>, Fn>
) =>
	fork(function* () {
		yield takeEvery(actionNames, saga, ...args);
		yield call(
			withDependencies(['subscriptionType'], function* ({ subscriptionType }: { subscriptionType: string }): SagaIterator {
				if (subscriptionType === SubscriptionType.platform) {
					yield bindActionsUntilStopped(actionNames);
				}
			}),
		);
	});

/**
 * Canceling the saga is behave the same as stop
 */
const bindActionsUntilStopped = (actionNames: string | string[]) =>
	fork(function* () {
		try {
			yield* bindActions(actionNames);
			yield take(StopApplication);
		} finally {
			yield* unbindActions(actionNames);
		}
	});

function* bindActions(actionNames: string | string[]) {
	actionNames = Array.isArray(actionNames) ? actionNames : [actionNames];
	for (const actionName of actionNames) {
		if (actionName !== '*') {
			yield put<BindAction>({
				type: BindAction,
				actionName,
			});
		}
	}
}

function* unbindActions(actionNames: string | string[]) {
	actionNames = Array.isArray(actionNames) ? actionNames : [actionNames];
	for (const actionName of actionNames) {
		if (actionName !== '*') {
			yield put<UnbindAction>({
				type: UnbindAction,
				actionName,
			});
		}
	}
}
