import { fetch } from '../../Isomorphic/fetch';
import { generateUniqueHash } from '@signageos/lib/dist/Hash/generator';
import { createDeferred } from '@signageos/lib/dist/Promise/deferred';
import ISocket, {
	IErrorEvent,
	IRequestMessage,
	IResponseMessage,
	UnconfirmedMessageError,
	UndeliveredEmitError,
} from '@signageos/lib/dist/WebSocket/Client/ISocket';
import Debug from 'debug';
import { EventEmitter } from 'events';
import { createProgressiveWait } from '@signageos/lib/dist/Timer/progressiveWait';
const debug = Debug('@signageos/front-display:Socket:Http:createHttpSocket');

enum Resource {
	SESSION = '/http-socket/session',
	TRANSMIT_MESSAGES = '/http-socket/transmit-messages',
	RECEIVE_MESSAGES = '/http-socket/receive-messages',
}

const DEFAULT_CHECK_INTERVAL = 60e3;

interface ISessionCredentials {
	/** unique session identification used to identify a single socket to server with time limited access */
	id: string;
	/** session secret used for verification of communication for session id */
	secret: string;
	/** timestamp in ms when the session expires */
	expireAt: number;
}

type IMessage = IRequestMessage | IResponseMessage;

export interface IOptions {
	messagesToServerQueue: IMessage[];
	checkInterval?: number;
}

export function createSocket(
	baseUrl: string,
	onConnected: () => void,
	onDisconnected: () => void,
	onError: (error: IErrorEvent | UndeliveredEmitError | UnconfirmedMessageError) => void,
	options: IOptions = {
		messagesToServerQueue: [],
	},
) {
	const maxCheckInterval = options.checkInterval ?? DEFAULT_CHECK_INTERVAL;
	const progressiveWait = createProgressiveWait(Math.min(1e3, maxCheckInterval), 1.5, maxCheckInterval);
	const messageEmitter = new EventEmitter();
	const responseEmitter = new EventEmitter();
	const messagesToServerQueue: IMessage[] = options.messagesToServerQueue;
	const terminationDeferred = createDeferred<void>();
	let closed = false;
	let sessionCredentials: ISessionCredentials | undefined;

	function emitUnconfirmedMessages() {
		let message: IMessage | undefined;
		while ((message = messagesToServerQueue.shift())) {
			debug('Undelivered message', message);
			if (message.type === 'request' && message.responseUid) {
				const responseMessage: IResponseMessage = { type: 'response', responseUid: message.responseUid };
				const error = new UnconfirmedMessageError(responseMessage, message, new Date());
				onError(error);
			}
		}
	}

	function getAuthHeaders(): Record<string, string> {
		if (sessionCredentials !== undefined) {
			return {
				'X-Auth': sessionCredentials.id + ':' + sessionCredentials.secret,
			};
		}
		throw new Error(`Session has not been initialized yet`);
	}

	async function createSession() {
		const sessionUri = baseUrl + Resource.SESSION;
		const response = await fetch(sessionUri, {
			method: 'POST',
		});
		if (!response.ok) {
			throw new Error(`Request for POST ${Resource.SESSION} failed with status code: ${response.status} - ${response.statusText}`);
		}
		const credentials: ISessionCredentials = await response.json();
		sessionCredentials = credentials;
	}

	async function removeSession() {
		if (sessionCredentials === undefined) {
			throw new Error(`Session has not been initialized yet`);
		}
		const sessionUri = baseUrl + Resource.SESSION + '/' + sessionCredentials.id;
		const response = await fetch(sessionUri, {
			method: 'DELETE',
			headers: getAuthHeaders(),
		});
		if (!response.ok) {
			throw new Error(
				`Request for DELETE ${Resource.SESSION}/${sessionCredentials.id} ` +
					`failed with status code: ${response.status} - ${response.statusText}`,
			);
		}
		sessionCredentials = undefined;
	}

	async function processMessagesToServer() {
		let message: IMessage | undefined;

		const messagesToTransmit: IMessage[] = [];
		while ((message = messagesToServerQueue.shift())) {
			messagesToTransmit.push(message);
		}
		try {
			const messagesUri = baseUrl + Resource.TRANSMIT_MESSAGES;
			const response = await fetch(messagesUri, {
				method: 'POST',
				headers: {
					...getAuthHeaders(),
					'Content-Type': 'application/json',
				},
				body: JSON.stringify(messagesToTransmit),
			});
			if (!response.ok) {
				throw new Error(
					`Request for POST ${Resource.TRANSMIT_MESSAGES} failed with status code: ${response.status} - ${response.statusText}`,
				);
			}
			for (const messageToTransmit of messagesToTransmit) {
				if (messageToTransmit.responseUid) {
					responseEmitter.emit(messageToTransmit.responseUid);
				}
			}
		} catch (error) {
			// Push all messages back to queue in the same order on error
			while ((message = messagesToTransmit.pop())) {
				messagesToServerQueue.unshift(message);
			}
			throw error;
		}
	}

	async function processMessagesFromServer() {
		const messagesUri = baseUrl + Resource.RECEIVE_MESSAGES;
		const response = await fetch(messagesUri, {
			method: 'POST',
			headers: getAuthHeaders(),
		});
		if (!response.ok) {
			throw new Error(`Request for POST ${Resource.RECEIVE_MESSAGES} failed with status code: ${response.status} - ${response.statusText}`);
		}
		const messages: IRequestMessage[] = await response.json();
		for (const message of messages) {
			messageEmitter.emit(message.event, message.payload);
			if (message.responseUid) {
				messagesToServerQueue.push({
					type: 'response',
					responseUid: message.responseUid,
				});
			}
		}
		if (messages.length > 0) {
			progressiveWait.reset();
		}
	}

	async function doRequesting() {
		try {
			await createSession();
			onConnected();
			while (!closed) {
				await processMessagesToServer();
				await processMessagesFromServer();
				await Promise.race([progressiveWait.wait(), terminationDeferred.promise]);
			}
		} finally {
			emitUnconfirmedMessages();
			onDisconnected();
			try {
				await removeSession();
			} catch (error) {
				// Clearing session is not required. Will not panic and continue.
				console.error(`Removing session failed`, error);
			}
		}
	}

	// Run requesting to HTTP in background (do not wait on result)
	doRequesting().catch((error: Error) => console.error(error));

	const socket: ISocket = {
		on(event: string, listener: (payload: any) => void) {
			messageEmitter.on(event, listener);
		},
		once(event: string, listener: (message: any) => void) {
			messageEmitter.once(event, listener);
		},
		emit(event: string, payload: any, callback?: () => void) {
			const message: IRequestMessage = { type: 'request', event, payload };
			if (closed) {
				debug('Socket is not open. Undelivered message', message);
				const error = new UndeliveredEmitError(message, new Date());
				onError(error);
				throw error;
			}
			if (callback) {
				message.responseUid = generateUniqueHash();
				responseEmitter.once(message.responseUid, () => callback());
			}
			messagesToServerQueue.push(message);
		},
		removeListener(event: string, listener: (payload: any) => void) {
			messageEmitter.removeListener(event, listener);
		},
		removeAllListeners() {
			messageEmitter.removeAllListeners();
		},
		close() {
			if (closed) {
				debug('Http socket is closed already');
				return;
			}
			closed = true;
			terminationDeferred.resolve();
		},
	};
	return socket;
}
