import { EventEmitter } from 'events';
import wait from '@signageos/lib/dist/Timer/wait';
import { IServletProcess, IServletProcessMessage } from './IServletRunner';
import Debug from 'debug';

const debug = Debug('@signageos/front-display:Servlet:SupervisedServletProcess');

/**
 * Servlet process wrapper that ensures that the process is always running
 *
 * If it stops, it restarts it.
 * Inspired by https://wiki.gentoo.org/wiki/OpenRC/supervise-daemon
 */
export default class SupervisedServletProcess implements IServletProcess {
	private servletProcess: IServletProcess;
	private eventEmitter: EventEmitter;
	private closed: boolean = false;

	constructor(private runProcess: () => IServletProcess) {
		this.eventEmitter = new EventEmitter();
		this.runAndSuperviseServletProcess();
	}

	public getId(): string {
		return this.servletProcess.getId();
	}

	public sendMessage<T extends IServletProcessMessage>(message: T): void {
		this.servletProcess.sendMessage(message);
	}

	public onMessage(messageListener: (message: IServletProcessMessage) => void): () => void {
		this.eventEmitter.addListener('message', messageListener);
		return () => this.eventEmitter.removeListener('message', messageListener);
	}

	public onceClosed(listener: () => void): void {
		this.eventEmitter.once('closed', listener);
	}

	public async stop(): Promise<void> {
		this.closed = true;
		await this.servletProcess.stop();
		this.eventEmitter.emit('closed');
	}

	private runAndSuperviseServletProcess() {
		debug('starting supervised servlet process');
		this.servletProcess = this.runProcess();
		this.servletProcess.onMessage((message: IServletProcessMessage) => {
			debug(`servlet ${this.servletProcess.getId()} message`, message);
			this.eventEmitter.emit('message', message);
		});
		this.servletProcess.onceClosed(async () => {
			debug(`servlet ${this.servletProcess.getId()} process was closed`);
			if (!this.closed) {
				const TIMEOUT_BEFORE_RESTART = 1e3;
				await wait(TIMEOUT_BEFORE_RESTART);
				this.runAndSuperviseServletProcess();
			}
		});
	}
}
