import { EventEmitter } from 'events';
import { SerialPortStream } from '@serialport/stream';
import { ISerialPort, SerialPortEvent } from '../NativeDevice/Hardware/ISerial';
import Debug from 'debug';
import { debug } from '@signageos/lib/dist/Debug/debugDecorator';

const DEBUG_NAMESPACE = '@signageos/front-display:Serial:SerialPort';
const logDebug = Debug(DEBUG_NAMESPACE);

export default class SerialPort implements ISerialPort {
	private eventEmitter: EventEmitter;

	constructor(private serialPort: SerialPortStream) {
		this.eventEmitter = new EventEmitter();
		this.bindAndConvertDataEvents();
	}

	public on(event: SerialPortEvent.DATA, listener: (data: Uint8Array) => void): void;
	public on(event: SerialPortEvent.CLOSE, listener: () => void): void;
	public on(event: SerialPortEvent.DATA | SerialPortEvent.CLOSE, listener: ((data: Uint8Array) => void) | (() => void)): void {
		this.eventEmitter.on(event, listener);
	}

	public removeListener(event: SerialPortEvent.DATA, listener: (data: Uint8Array) => void): void;
	public removeListener(event: SerialPortEvent.CLOSE, listener: () => void): void;
	public removeListener(event: SerialPortEvent.DATA | SerialPortEvent.CLOSE, listener: ((data: Uint8Array) => void) | (() => void)): void {
		this.eventEmitter.removeListener(event, listener);
	}

	public removeAllListeners(event?: SerialPortEvent): void {
		if (event) {
			this.eventEmitter.removeAllListeners(event);
		} else {
			this.eventEmitter.removeAllListeners();
		}
	}

	@debug(DEBUG_NAMESPACE)
	public write(data: string | number[] | Uint8Array): Promise<void> {
		return new Promise((resolve: () => void, reject: (error: Error) => void) => {
			let convertedData: string | number[] | Buffer;
			if (data instanceof Uint8Array) {
				convertedData = Buffer.from(data);
			} else {
				convertedData = data;
			}
			this.serialPort.write(convertedData, (error?: Error) => {
				if (error) {
					reject(error);
				} else {
					resolve();
				}
			});
		});
	}

	@debug(DEBUG_NAMESPACE)
	public async close(): Promise<void> {
		if (this.serialPort.isOpen) {
			await this.drainSocket();
			await this.closeSocket();
		}
	}

	private drainSocket() {
		return new Promise((resolve: (value?: unknown) => void, reject: (error: Error) => void) => {
			this.serialPort.drain((error: Error | null) => {
				if (error) {
					reject(error);
				} else {
					resolve();
				}
			});
		});
	}

	private closeSocket() {
		return new Promise((resolve: (value?: unknown) => void, reject: (error: Error) => void) => {
			this.serialPort.close((error: Error | null) => {
				if (error) {
					reject(error);
				} else {
					resolve();
				}
			});
		});
	}

	private bindAndConvertDataEvents() {
		this.serialPort.on('data', (data: string | Buffer) => {
			logDebug(this.serialPort.path + ' data', data.toString());
			if (!Buffer.isBuffer(data)) {
				this.eventEmitter.emit('data', Buffer.from(data));
			} else {
				this.eventEmitter.emit('data', data);
			}
		});
		this.serialPort.on('close', () => {
			logDebug(this.serialPort.path + ' closed');
			this.eventEmitter.emit('close');
		});
	}
}
