import { call, fork, spawn } from 'redux-saga/effects';

export interface IChannel<TMessage> {
	take(): Promise<TMessage>;
	put(message: TMessage): void;
	close(): void;
}

export const takeEvery = function* <TMessage>(channel: IChannel<TMessage>, saga: (action: TMessage) => any) {
	yield spawn(function* () {
		while (true) {
			const message: TMessage = yield call(channel.take);
			yield fork(saga, message);
		}
	});
};

export function createChannel<TMessage>(executor?: (put: (message: TMessage) => void) => void): IChannel<TMessage> {
	const messageQueue: TMessage[] = [];
	const resolveQueue: ((message: TMessage) => void)[] = [];
	const rejectQueue: (() => void)[] = [];
	let open = true;

	function put(message: TMessage): void {
		if (resolveQueue.length) {
			const nextResolve = resolveQueue.shift()!;
			rejectQueue.shift();
			nextResolve(message);
		} else if (open) {
			messageQueue.push(message);
		}
	}

	function take(): Promise<TMessage> {
		if (messageQueue.length) {
			return Promise.resolve(messageQueue.shift()!);
		} else if (open) {
			return new Promise((resolve: (message: TMessage) => void, reject: () => void) => {
				resolveQueue.push(resolve);
				rejectQueue.push(reject);
			});
		} else {
			return Promise.reject();
		}
	}

	function close(): void {
		open = false;
		while (resolveQueue.length > 0) {
			resolveQueue.shift();
			const nextReject = rejectQueue.shift()!;
			nextReject();
		}
	}

	if (executor) {
		executor(put);
	}

	return {
		take,
		put,
		close,
	};
}
