import * as dgram from 'dgram';
import { EventEmitter } from 'events';
import { IUdpSocket, UdpSocketEvent } from './IUdpSocket';

const BROADCAST_IP = '255.255.255.255';

/**
 * UDP socket implementation.
 *
 * Handles sending and receiving messages via UDP.
 */
export class UdpSocket implements IUdpSocket<Buffer> {
	private emitter: EventEmitter = new EventEmitter();
	private socket: dgram.Socket;

	constructor(private port: number) {
		this.handleMessage = this.handleMessage.bind(this);
		this.socket = this.createSocket();
	}

	public async open(): Promise<void> {
		await new Promise<void>((resolve: () => void) => {
			this.socket.once('listening', () => {
				this.socket.setBroadcast(true);
				resolve();
			});
		});
	}

	public async close(): Promise<void> {
		await new Promise<void>((resolve) => {
			this.socket.close(resolve);
		});

		this.socket.removeAllListeners();
	}

	public async send(messageBuffer: Buffer): Promise<void> {
		await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			this.socket.send(messageBuffer, 0, messageBuffer.length, this.port, BROADCAST_IP, (error: Error | null) => {
				if (error) {
					reject(error);
				} else {
					resolve();
				}
			});
		});
	}

	public addListener(event: UdpSocketEvent, callback: (message: Buffer) => void): void {
		this.emitter.addListener(event, callback);
	}

	public removeListener(event: UdpSocketEvent, callback: (message: Buffer) => void): void {
		this.emitter.removeListener(event, callback);
	}

	private createSocket() {
		const socket = dgram.createSocket({
			type: 'udp4',
			reuseAddr: true,
		});

		socket.addListener('message', this.handleMessage);
		socket.bind(this.port);

		return socket;
	}

	private handleMessage(messageBuffer: Buffer) {
		this.emitter.emit(UdpSocketEvent.Message, messageBuffer);
	}
}
