import { EventEmitter } from 'events';
import { ISerializer } from '../../Serializer/serialization';
import { IUdpSocket, MessageListener, UdpSocketEvent } from './IUdpSocket';

export class MessageUdpSocket<TSerialized> implements IUdpSocket<object> {
	private emitter: EventEmitter = new EventEmitter();

	constructor(
		private socket: IUdpSocket<TSerialized>,
		private serializer: ISerializer<TSerialized>,
	) {
		this.handleSocketMessage = this.handleSocketMessage.bind(this);
		this.listenToSocket();
	}

	public async close(): Promise<void> {
		await this.socket.close();
		this.socket.removeListener(UdpSocketEvent.Message, this.handleSocketMessage);
	}

	public async send(message: object): Promise<void> {
		const serializedMessage = this.serializer.serialize(message);
		await this.socket.send(serializedMessage);
	}

	public addListener(event: UdpSocketEvent, callback: MessageListener<object>): void {
		this.emitter.addListener(event, callback);
	}

	public removeListener(event: UdpSocketEvent, callback: MessageListener<object>): void {
		this.emitter.removeListener(event, callback);
	}

	private listenToSocket() {
		this.socket.addListener(UdpSocketEvent.Message, this.handleSocketMessage);
	}

	private handleSocketMessage(serializedMessage: TSerialized) {
		const message = this.serializer.deserialize(serializedMessage);
		this.emitter.emit(UdpSocketEvent.Message, message);
	}
}
