import { EventEmitter } from 'events';
import { ClosedListener, ITcpSocket, MessageListener, SendArgs, TcpSocketEvent, TcpSocketEventListener } from './ITcpSocket';
import { TcpClient } from './TcpClient';
import { TcpServer, TcpServerEvent } from './TcpServer';

export interface TcpSocketConfig {
	/** Server will listen on this port */
	port: number;
	/** Client connection will be only allowed to stay alive for this long. After that it will be killed. */
	clientConnectionTTLMs?: number;
	/** Client will only try for this long to establish a connection to a server. After that it will give up and throw an error. */
	clientConnectionTimeoutMs?: number;
}

export class TcpSocket implements ITcpSocket<Buffer> {
	private emitter: EventEmitter = new EventEmitter();
	private server: TcpServer;

	constructor(private config: TcpSocketConfig) {
		this.server = new TcpServer(config.port, config.clientConnectionTTLMs);
		this.listenToServerEvents();
	}

	public async open() {
		await this.server.open();
	}

	public async close(): Promise<void> {
		await this.server.close();
	}

	public async send(args: SendArgs<Buffer>): Promise<void> {
		const client = new TcpClient({
			host: args.host,
			port: args.port,
			connectionTimeoutMs: this.config.clientConnectionTimeoutMs,
		});

		await client.connect();
		try {
			await client.send(args.message);
		} finally {
			await client.close();
		}
	}

	public addListener(event: TcpSocketEvent.Message, callback: MessageListener<Buffer>): void;
	public addListener(event: TcpSocketEvent.Closed, callback: ClosedListener): void;
	public addListener(event: TcpSocketEvent, callback: TcpSocketEventListener<Buffer>): void {
		this.emitter.addListener(event, callback);
	}

	public removeListener(event: TcpSocketEvent.Message, callback: MessageListener<Buffer>): void;
	public removeListener(event: TcpSocketEvent.Closed, callback: ClosedListener): void;
	public removeListener(event: TcpSocketEvent, callback: TcpSocketEventListener<Buffer>): void {
		this.emitter.removeListener(event, callback);
	}

	private listenToServerEvents() {
		this.server.addListener(TcpServerEvent.Message, (message) => {
			this.handleMessage(message);
		});

		this.server.addListener(TcpServerEvent.Closed, (error) => {
			this.emitter.emit(TcpSocketEvent.Closed, error);
		});
	}

	private handleMessage(message: Buffer) {
		this.emitter.emit(TcpSocketEvent.Message, message);
	}
}
