import { EventEmitter } from 'events';
import * as net from 'net';

export enum TcpServerEvent {
	Message = 'message',
	Closed = 'closed',
}

const DEFAULT_CLIENT_CONNECTION_TTL_MS = 5e3;

export type TcpServerMessageListener = (message: Buffer) => void;
export type TcpServerClosedListener = (error?: Error) => void;
export type TcpServerEventListener = TcpServerMessageListener | TcpServerClosedListener;

export class TcpServer {
	private emitter: EventEmitter = new EventEmitter();
	private server: net.Server;

	constructor(
		private port: number,
		private clientConnectionTTLMs: number = DEFAULT_CLIENT_CONNECTION_TTL_MS,
	) {
		this.server = new net.Server();
		this.listenToMessagesFromClients();
	}

	public async open(): Promise<void> {
		await this.openServer();
		this.closeServerAndEmitClosedEventOnError();
	}

	public async close(): Promise<void> {
		await this.closeServer();
	}

	public addListener(event: TcpServerEvent.Message, listener: TcpServerMessageListener): void;
	public addListener(event: TcpServerEvent.Closed, listener: TcpServerClosedListener): void;
	public addListener(event: TcpServerEvent, listener: TcpServerEventListener) {
		this.emitter.addListener(event, listener);
	}

	public removeListener(event: TcpServerEvent.Message, listener: TcpServerMessageListener): void;
	public removeListener(event: TcpServerEvent.Closed, listener: TcpServerClosedListener): void;
	public removeListener(event: TcpServerEvent, listener: TcpServerEventListener) {
		this.emitter.removeListener(event, listener);
	}

	private listenToMessagesFromClients() {
		this.server.on('connection', (socket) => {
			this.handleConnection(socket);
		});
	}

	private handleConnection(socket: net.Socket) {
		const onDataListener = (data: Buffer) => {
			this.handleMessage(data);
		};

		// prevent rogue clients from keeping the connection alive forever
		const timeout = setTimeout(() => socket.destroy(), this.clientConnectionTTLMs);

		socket.on('data', onDataListener);
		socket.on('close', () => {
			socket.removeListener('data', onDataListener);
			clearTimeout(timeout);
		});
	}

	private handleMessage(message: Buffer) {
		this.emitter.emit(TcpServerEvent.Message, message);
	}

	private openServer() {
		return new Promise<void>((resolve, reject) => {
			const cleanupListeners = () => {
				this.server.removeAllListeners('listening');
				this.server.removeAllListeners('error');
			};

			// cleanup potential listeners from previous runs
			cleanupListeners();

			this.server.once('listening', () => {
				cleanupListeners();
				resolve();
			});
			this.server.once('error', (error) => {
				cleanupListeners();
				reject(error);
			});

			this.server.listen(this.port);
		});
	}

	private closeServerAndEmitClosedEventOnError() {
		this.server.once('error', (error) => {
			this.closeServer(error);
		});
	}

	private closeServer(error?: Error) {
		return new Promise<void>((resolve) => {
			this.server.close((closeError) => {
				if (closeError) {
					console.warn('TcpServer closed with error', closeError);
				}

				this.emitter.emit(TcpServerEvent.Closed, error);
				resolve();
			});
		});
	}
}
